#include "parse_bs_watch_log.h"

#include <crypta/graph/rtmr/proto/yuid_message.pb.h>
#include <crypta/graph/rtmr/lib/common/browserinfo.h>
#include <crypta/graph/rtmr/lib/common/constants.h>
#include <crypta/graph/rtmr/lib/common/deduplicator_indexes.h>
#include <crypta/graph/rtmr/lib/common/get_normalized_host.h>
#include <crypta/graph/rtmr/lib/common/get_source_by_ip.h>
#include <crypta/graph/rtmr/lib/common/make_parsed_bs_watch_row.h>
#include <crypta/graph/rtmr/lib/common/normalize_ip.h>
#include <crypta/graph/rtmr/lib/common/serialize_fingerprint_match.h>
#include <crypta/graph/rtmr/lib/common/validate.h>
#include <crypta/lib/native/fingerprint/fingerprint.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/ext_fp/constants.h>
#include <crypta/lib/native/url/url_utils.h>
#include <crypta/lib/proto/ext_fp/fp_event.pb.h>
#include <util/draft/ip.h>

#include <library/cpp/string_utils/base64/base64.h>

using namespace NCrypta;
using namespace NCrypta::NGraph;

namespace {
    bool CheckYsclidTimeframe(const TString& ysclidValue, const TString& clientTimeValue) {
        static const ui64 clickTimeFrame{20};

        const NIdentifiers::TYSClid ysclid{ysclidValue};

        if (!ysclid.IsSignificant()) {
            // incorect ysclid format
            return false;
        }

        ui64 clientTime{0};
        if (!TryFromString<ui64>(clientTimeValue, clientTime)) {
            // incorect browserinfo eventtime format
            return false;
        }

        const ui64 ysclidTime{ysclid.GetTimestamp()};
        return (clientTime >= ysclidTime)
            && (clientTime - ysclidTime < clickTimeFrame)
        ;
    }
}

namespace {
    struct SyncedResult {
        TString Duid;
        TString Domain;
    };

    TMaybe<SyncedResult> ExtractSyncDuid(const TString& browserInfo) {
        if (browserInfo.empty()) {
            return Nothing();
        }

        auto duid = NBrowserInfo::GetValue(browserInfo, "pi");
        auto domain = Base64DecodeUneven(NBrowserInfo::GetValue(browserInfo, "pid"));

        if (!IsValidIdentifier<NIdentifiers::TDuid>(duid)) {
            return Nothing();
        }

        return SyncedResult{
            .Duid = std::move(duid),
            .Domain = domain.empty() ? "auto.ru" : domain,
        };
    }

    bool DoCommonChecks(
        const TString& browserInfo,
        const TString& counterId,
        const uatraits::detector::result_type& traits,
        const TString& domainUserId) {
        return ValidateBrowserInfo(browserInfo) &&
               ValidateCounterID(counterId) &&
               ValidateDomainUserID(domainUserId) &&
               ValidateUATraits(traits);
    }
}

TString TParseBsWatchLog::SelectIP(const NYT::TNode& node) {
    const auto& ip4 = GetSafe<TString>(node, "clientip");
    if (ip4.empty() || ip4 == "0.0.0.0") {
        return GetSafe<TString>(node, "clientip6");
    }

    return ip4;
}

TParseBsWatchLog::TParseBsWatchLog(const TResources& resources, const NGeobase::TLookup& geoData)
    : GeoData(geoData)
    , Resources(resources)
{
    TShiftedClock::FreezeTimestampFromEnv();
}

void TParseBsWatchLog::DoParsed(const NYT::TNode& parsed, TWriter* writer) {
    const static char yandexCounterId[] = "722545";
    const static TString yclidParamName = "yclid";
    const static TString ysclidParamName = "ysclid";

    const auto& browserInfo = GetSafe<TString>(parsed, "browserinfo");
    const auto& counterId = GetSafe<TString>(parsed, "counterid");
    const auto& domainUserId = GetSafe<TString>(parsed, "domainuserid");
    const auto& referer = GetSafe<TString>(parsed, "referer");
    const auto& unixtime = GetSafe<TString>(parsed, "unixtime");
    const auto& url = GetSafe<TString>(parsed, "url");
    const auto& userAgent = GetSafe<TString>(parsed, "useragent");
    const auto& yuid = GetSafe<TString>(parsed, "uniqid", "0");
    const auto& clientIp6 = GetSafe<TString>(parsed, "clientip6", "::");

    TResources::TResourcesHolder holder(Resources.GetHolder());
    const auto& detector = holder.GetUatraitsDetector();
    const auto& sources = holder.GetSources();

    try {
        const auto& traits = detector.detect(userAgent);
        const auto& originalDomain = GetNormalizedHost(url);

        const bool areCommonChecksPassed{DoCommonChecks(browserInfo, counterId, traits, domainUserId)};
        const bool isGoodDomain{ValidateDomain(originalDomain)};
        const bool isValidYuid{ValidateYandexuid(yuid)};
        const bool refererFromYandex{ValidateYandexReferer(referer, originalDomain)};

        const auto timestamp = FromString<i64>(unixtime);

        if (IsGDPR(GeoData, GetSafe<TString>(parsed, "regionid"))) {
            return;
        }

        // zen & fp
        auto urlWithZenUtmReferer = url.EndsWith(NConstants::ZenRefererPostfix);
        if ((refererFromYandex || urlWithZenUtmReferer) && areCommonChecksPassed && isGoodDomain && !isValidYuid && !IsPrivateRelayIp(GeoData, clientIp6)) {
            const auto& parsedBsWatchRow = MakeBsWatchRow(domainUserId, *originalDomain, traits, timestamp, "", "fp");
            const auto& key = NFingerprint::CalculateFingerprint(url, Ip4Or6FromString(SelectIP(parsed).c_str()), userAgent);
            writer->AddRow({key, unixtime, MakeParsedBsWatchRow(parsedBsWatchRow)}, EOutputTo::JoinRedir);
        }

        // watch
        if (isValidYuid && areCommonChecksPassed && isGoodDomain && ShouldSaveYuid(browserInfo)) {
            const auto& parsedBsWatchRow = MakeBsWatchRow(domainUserId, *originalDomain, traits, timestamp, "", "watch");
            const auto& value = SerializeMatch(yuid, parsedBsWatchRow, "watch");

            writer->AddRow({yuid, unixtime, value}, EOutputTo::Logbroker);
        }

        // cookie_sync
        if (auto result = ExtractSyncDuid(browserInfo);
            result && isValidYuid && counterId == yandexCounterId) {
            const auto& parsedBsWatchRow = MakeBsWatchRow(result->Duid, result->Domain, traits, timestamp, "", "cookie_sync");
            const auto& value = SerializeMatch(yuid, parsedBsWatchRow, "cookie_sync");

            writer->AddRow({yuid, unixtime, value}, EOutputTo::Logbroker);
        }

        // tls
        const auto& sessionTicket = GetSafe<TString>(parsed, "sslsessionticketiv");
        if (areCommonChecksPassed && originalDomain.Defined() && ValidateSslSessionTicketIV(sessionTicket)) {
            const auto& parsedBsWatchRow = MakeBsWatchRow(domainUserId, *originalDomain, traits, timestamp, isValidYuid ? yuid : "", "tls");
            writer->AddRow({sessionTicket, unixtime, MakeParsedBsWatchRow(parsedBsWatchRow)}, EOutputTo::JoinSsl);
        }

        // yclid or ysclid
        const auto addYclidOrYSclid{
            [&](const TString& paramName, const auto& validateParam, const auto& outputTable) {
                const auto& value{GetParam(url, paramName)};
                if (areCommonChecksPassed && isGoodDomain && value.Defined() && validateParam(*value)) {
                    const auto& parsedBsWatchRow{MakeBsWatchRow(domainUserId, *originalDomain, traits, timestamp, "", paramName)};
                    writer->AddRow({*value, unixtime, MakeParsedBsWatchRow(parsedBsWatchRow)}, outputTable);
                }
            }
        };
        addYclidOrYSclid(yclidParamName, ValidateYClid, EOutputTo::JoinYClid);
        addYclidOrYSclid(ysclidParamName,
                         [&](const auto& ysclid) {
                            return
                                refererFromYandex
                                && CheckYsclidTimeframe(ysclid, NBrowserInfo::GetValue(browserInfo, "et"));
                         },
                         EOutputTo::JoinYSClid);

        // extfp
        auto fpSource = FindFirstPassing(sources, clientIp6, GetSourceByIpSafe(GeoData, clientIp6));
        bool fpChecksPassed = fpSource
            && originalDomain.Defined()
            && ValidateBrowserInfo(browserInfo)
            && ValidateCounterID(counterId)
            && ValidateDomainUserID(domainUserId)
            && (!fpSource->MatchItpOnly() || ValidateUATraits(traits));

        if (fpChecksPassed) {
            const static TString sourcePrefix{"watchlog:"};
            NExtFp::TFpEvent event;

            event.SetLogType("bs-watch-log");
            event.SetWatchId(FromString<ui64>(GetSafe<TString>(parsed, "watchid", "0")));
            event.SetIp(NormalizeIp(clientIp6).data());
            event.SetPort(FromString<ui32>(GetSafe<TString>(parsed, "tcpport", "0")));
            event.SetDuid(FromString<ui64>(domainUserId));
            event.SetUnixtime(timestamp);
            if (isValidYuid) { event.SetYuid(FromString<ui64>(yuid)); }
            event.SetUserAgent(userAgent);
            event.SetDomain(*originalDomain);
            event.SetCurrentTimestamp(TShiftedClock::Now().Seconds());
            event.SetSourceId(fpSource->GetSource());

            const auto outputTableIndex = ToString(static_cast<int>(fpSource->IsDelayed()
                ? EDeduplicatorTableIndex::ToExtFpDelayed
                : EDeduplicatorTableIndex::ToExtFp));
            writer->AddRow({sourcePrefix + domainUserId, outputTableIndex, NProtoSerializer::ToJson(event)},
                EOutputTo::ToDeduplicator);
        }

    } catch (const yexception& e) {
        writer->AddRow({yuid, unixtime, e.what()}, EOutputTo::Errors);
    }
}
