#include "parse_extfp_match_log.h"

#include <crypta/graph/rtmr/lib/common/validate.h>
#include <crypta/graph/rtmr/lib/common/make_parsed_bs_watch_row.h>
#include <library/cpp/json/json_reader.h>
#include <util/digest/city.h>
#include <util/string/split.h>

namespace {
    static const TString KEY_DOMAIN{"original_domain"};
    static const TString KEY_DUID{"duid"};
    static const TString KEY_EXT_ID{"ext_id"};
    static const TString KEY_EXT_SOURCE{"ext_source"};
    static const TString KEY_UA{"user_agent"};
    static const TString KEY_UNIXTIME{"unixtime"};
    static const TString KEY_YANDEXUID{"yuid"};
}

namespace NCrypta::NGraph {
    TParseExtfpMatchLog::TParseExtfpMatchLog(const TResources& resources)
        : Resources(resources)
    {
    }

    TString TParseExtfpMatchLog::CreateJoinKey(const TString& sourceId, const TString& extfp, const TString& userAgent) const {
        auto digest = CityHash64(TStringBuilder() << sourceId << extfp << userAgent);

        return ToString(digest);
    }

    void TParseExtfpMatchLog::ParseAndWrite(const TReader::TRowType& row,
                                       const uatraits::detector& detector,
                                       TWriter* writer) {

        for (const auto& it : StringSplitter(row.Value).Split('\n').SkipEmpty()) {
            try {
                NJson::TJsonValue parsed;
                NJson::ReadJsonTree(it.Token(), &parsed, true);

                const auto& domain = parsed[KEY_DOMAIN].GetString();
                const auto& duid = ToString(parsed[KEY_DUID].GetInteger());
                const auto& ext_id = parsed[KEY_EXT_ID].GetString();
                const auto& ext_source = parsed[KEY_EXT_SOURCE].GetString();
                const auto& user_agent = parsed[KEY_UA].GetString();
                const auto unixtime = parsed[KEY_UNIXTIME].GetInteger();
                const auto& yuid = ToString(parsed[KEY_YANDEXUID].GetInteger());

                const auto& traits = detector.detect(user_agent);

                auto parsedBsWatchRow = MakeBsWatchRow(duid, domain, traits, unixtime, ValidateYandexuid(yuid) ? yuid : "", ext_source);

                const auto joinKey = CreateJoinKey(ext_source, ext_id, user_agent);
                writer->AddRow({joinKey, ToString(unixtime), MakeParsedBsWatchRow(parsedBsWatchRow)}, EOutputTable::JoinExtfpAndUa);

            } catch(const yexception& e) {
                writer->AddRow({row.Key, row.SubKey, e.what()}, Errors);
            }
        }
    }

    void TParseExtfpMatchLog::Do(TReader* reader, TWriter* writer) {
        auto resourceHolder = Resources.GetHolder();
        const auto& detector = resourceHolder.GetUatraitsDetector();

        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();

            ParseAndWrite(row, detector, writer);
        }
    }
}
