#include "parse_redir_log.h"

#include <library/cpp/yson/node/node_io.h>
#include <crypta/graph/rtmr/lib/common/get_normalized_host.h>
#include <crypta/graph/rtmr/lib/common/validate.h>
#include <crypta/lib/native/fingerprint/fingerprint.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>

#include <util/draft/ip.h>

using namespace NYT;

namespace {
    template <typename T>
    T GetSafe(const TNode& node, const TString& key) {
        if (const auto& resultNode{GetSafe<TNode>(node, key)};
            resultNode.HasValue()) {
            return resultNode.As<T>();
        }
        return {};
    }

    template <>
    TNode GetSafe<TNode>(const TNode& node, const TString& key) {
        if (!node.HasValue()) {
            return {};
        }
        const auto& nodeMap = node.AsMap();
        const auto& it = nodeMap.find(key);

        return (it != nodeMap.end()) ? it->second : TNode{};
    }
}

namespace NCrypta::NGraph {
    void TTskvParser::Parse(TStringBuf rawRecord, TNode& parsed) {
        ParseRaw(rawRecord, parsed);
    }

    TParseRedirLog::TParseRedirLog(const TResources& resources)
        : Resources(resources)
    {
    }

    void TParseRedirLog::ParseAndWrite(const TReader::TRowType& row,
                                       const uatraits::detector& detector,
                                       TWriter* writer) {
        TNode parsed;

        try {
            Parser.Parse(row.Value, parsed);

            TString yuid;
            if (row.Key.StartsWith("y")) {
                yuid = row.Key.SubString(1, row.Key.Size() - 1);
            }

            if (yuid.empty()) {
                return;
            }

            const auto& url = GetSafe<TString>(parsed, "url");
            const auto& ip = GetSafe<TString>(parsed, "ip");
            const auto& userAgent = GetSafe<TString>(parsed, "u-agent");

            const auto& traits = detector.detect(userAgent);
            const auto& domain = GetNormalizedHost(url);

            if (!ValidateYandexuid(yuid) || !ValidateDomain(domain) || !ValidateUATraits(traits)) {
                return;
            }

            // fingerprint message
            {
                const auto& fingerprint = NFingerprint::CalculateFingerprint(url, Ip4Or6FromString(ip.c_str()), userAgent);
                TYuidMessage message;
                message.SetYandexuid(yuid);
                message.SetTimestamp(FromString<i64>(row.SubKey));
                message.SetSource("serp");

                writer->AddRow({fingerprint, row.SubKey, NProtoSerializer::ToString(message)}, OutputFp);
            }

            // ysclid message
            if (const TString events{GetSafe<TString>(parsed, "events")}; !events.empty()) {
                TParsedBsWatchRow message;
                message.SetYuid(yuid);
                message.SetTimestamp(FromString<i64>(row.SubKey));
                message.SetSource("ysclid");

                const auto& parsedJson{NodeFromJsonString(events)};
                for (const TNode& item : parsedJson.AsList()) {
                    const auto& eventData{GetSafe<TNode>(item, "data")};
                    const auto& ysclid{GetSafe<TString>(eventData, "ysclid")};
                    if (NIdentifiers::TYSClid::Significate(ysclid)) {
                        writer->AddRow({ysclid, row.SubKey, NProtoSerializer::ToString(message)}, OutputYSclid);
                    }
                }
            }

        } catch (const yexception& e) {
            writer->AddRow({row.Key, row.SubKey, e.what()}, Errors);
        }
    }

    void TParseRedirLog::Do(TReader* reader, TWriter* writer) {
        auto resourceHolder = Resources.GetHolder();
        const auto& detector = resourceHolder.GetUatraitsDetector();

        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();

            ParseAndWrite(row, detector, writer);
        }
    }
}
