#include "parse_zen_log.h"
#include <crypta/graph/rtmr/lib/common/constants.h>
#include <crypta/graph/rtmr/lib/common/get_normalized_host.h>
#include <crypta/graph/rtmr/lib/common/validate.h>
#include <crypta/lib/native/fingerprint/fingerprint.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <library/cpp/json/json_reader.h>

namespace {
    static const TString KEY_ITEMTYPE{"item_type"};
    static const TString KEY_URL{"url"};
    static const TString KEY_YANDEXUID{"yandexuid"};
    static const TString KEY_UA{"user_agent"};
    static const TString KEY_INTERNAL{"yandex_internal"};
    static const TString KEY_IP{"ip"};
}

namespace NCrypta::NGraph {
    TParseZenLog::TParseZenLog(const TResources& resources)
        : Resources(resources)
    {
    }

    void TParseZenLog::ParseAndWrite(const TReader::TRowType& row,
                                     const uatraits::detector& detector,
                                     TWriter* writer) {
        try {
            NJson::TJsonValue parsed;
            NJson::ReadJsonTree(row.Value, &parsed);

            const auto& yuid = parsed[KEY_YANDEXUID].GetString();
            const auto& url = parsed[KEY_URL].GetString();
            const auto& ip = parsed[KEY_IP].GetString();
            const auto& ua = parsed[KEY_UA].GetString();

            if (parsed[KEY_ITEMTYPE].GetString() != "url" || parsed[KEY_URL].GetString().empty() || parsed[KEY_INTERNAL].GetBoolean() || !ValidateYandexuid(yuid) || !ValidateAddress(ip)) {
                return;
            }

            const auto& traits = detector.detect(ua);
            const auto& domain = GetNormalizedHost(url);
            if (!ValidateUATraits(traits) || !ValidateDomain(domain)) {
                return;
            }

            TYuidMessage message;
            message.SetYandexuid(yuid);
            message.SetTimestamp(FromString<i64>(row.SubKey));
            message.SetSource("zen");

            const auto& ipString = Ip4Or6FromString(ip.c_str());
            const auto& fingerprint = NFingerprint::CalculateFingerprint(url, ipString, ua);
            const auto& fingerprintPatched = NFingerprint::CalculateFingerprint(url + NConstants::ZenRefererPostfix, ipString, ua);
            const auto& messageSerialized = NProtoSerializer::ToString(message);

            writer->AddRow({fingerprint, row.SubKey, messageSerialized}, Output);
            writer->AddRow({fingerprintPatched, row.SubKey, messageSerialized}, Output);

        } catch (const yexception& e) {
            writer->AddRow({row.Key, row.SubKey, e.what()}, Errors);
        }
    }

    void TParseZenLog::Do(TReader* reader, TWriter* writer) {
        auto resourceHolder = Resources.GetHolder();
        const auto& detector = resourceHolder.GetUatraitsDetector();

        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();

            ParseAndWrite(row, detector, writer);
        }
    }

}
