#include "tls_reducer.h"
#include <crypta/graph/rtmr/lib/common/serialize_fingerprint_match.h>
#include <crypta/graph/rtmr/lib/common/validate.h>
#include <crypta/lib/native/identifiers/lib/id_types/yandexuid.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <util/generic/hash_set.h>

using namespace NCrypta::NGraph;

namespace {
    bool isInRepeated(const ::google::protobuf::RepeatedPtrField<TParsedBsWatchRow>& field, const TString& value) {
        return std::find_if(field.begin(), field.end(), [&value](const auto& v) { return v.GetFpc() == value; }) != field.end();
    }
}

bool TTlsReducer::IsNewYandexuidBetter(const TString& current, const TString& candidate) {
    using NIdentifiers::TYandexuid;
    return !candidate.empty() && (current.empty() || (TYandexuid::GetTimestamp(candidate) < TYandexuid::GetTimestamp(current)));
}

TTlsReducer::EDecision TTlsReducer::CheckAndModifyState(const TParsedBsWatchRow& row) {
    bool yandexuidChanged = false;
    bool duidNew = false;
    const auto& stateYuid = State.GetYuid();

    const auto& duid = row.GetFpc();
    const auto& yuid = row.GetYuid();

    bool isDomainUploadable = ValidateDomain(row.GetDomain());

    /* first check if duid've been seen and remember it */
    if (isDomainUploadable && stateYuid.empty() && State.GetDuids().size() < MaxAllowedDuids) {
        duidNew = !isInRepeated(State.GetDuids(), duid);
        if (duidNew) {
            State.MutableDuids()->Add()->CopyFrom(row);
        }
    } else if (isDomainUploadable && State.GetUploadedDuids().size() < MaxAllowedDuids) {
        duidNew = !isInRepeated(State.GetUploadedDuids(), duid);
        if (duidNew) {
            State.MutableUploadedDuids()->Add()->CopyFrom(row);
        }
    }

    /* now check if yandexuid had been updated */
    yandexuidChanged = IsNewYandexuidBetter(stateYuid, yuid);
    if (yandexuidChanged) {
        State.SetYuid(yuid);
        for (auto& message : *State.MutableDuids()) {
            State.MutableUploadedDuids()->Add()->Swap(&message);
        }
        State.MutableDuids()->Clear();
    }

    if (yandexuidChanged) {
        return EDecision::UploadAll;
    }
    if (duidNew && !stateYuid.empty()) {
        return EDecision::UploadOnce;
    }
    return EDecision::Skip;
}

void TTlsReducer::Do(TReader* reader, TWriter* writer) {
    TShiftedClock::FreezeTimestampFromEnv();

    for (; reader->IsValid(); reader->Next()) {
        const auto& row = reader->GetRow();

        try {
            TParsedBsWatchRow message;
            Y_PROTOBUF_SUPPRESS_NODISCARD message.ParseFromString(TString{row.Value});
            const TString& source = message.GetSource().empty() ? "tls" : message.GetSource();

            switch (CheckAndModifyState(message)) {
                case EDecision::UploadAll: {
                    const auto& yuid = State.GetYuid();
                    for (const auto& message : State.GetUploadedDuids()) {
                        writer->AddRow({row.Key, row.SubKey, SerializeMatch(yuid, message, source)}, EOutputTo::Logbroker);
                    }
                    break;
                }
                case EDecision::UploadOnce: {
                    writer->AddRow({row.Key, row.SubKey, SerializeMatch(State.GetYuid(), message, source)}, EOutputTo::Logbroker);
                    break;
                }
                default:
                    break;
            }
        } catch (const yexception& e) {
            writer->AddRow({row.Key, row.SubKey, e.what()}, EOutputTo::Errors);
        }
    }
}
