import logging
import yt.wrapper as yt

from cached_property import cached_property

from crypta.lib.python.yql_runner.base_parser import BaseParser

logging.basicConfig(level=logging.INFO, format='%(asctime)s %(message)s')
logger = logging.getLogger(__name__)


class Sampler(BaseParser):
    """ Run YQL query to sampling soup edges for given percent of MRCC-components """
    COOKED_SOUP_EDGES_PATH = yt.ypath.ypath_join(BaseParser.GRAPH_OUTPUT_DIR, "v2/soup/cooked/soup_edges")
    COOKED_SOUP_VERTICES_PROPERTIES_PATH = yt.ypath.ypath_join(BaseParser.GRAPH_OUTPUT_DIR, "v2/soup/cooked/vertices_properties")

    GENERATE_DATE_ATTR = "generate_date"

    def __init__(self, components_table, output_edges, output_properties, yt_proxy=None, yt_pool="crypta", is_embedded=False, **kwargs):
        yt.config.set_proxy(yt_proxy)
        if not yt.exists(components_table):
            raise Exception('Table "{table}" does not exist'.format(table=components_table))

        self.components_table = components_table
        self.output_edges = output_edges
        self.output_properties = output_properties

        super(Sampler, self).__init__(
            date=None,
            yt_proxy=yt_proxy,
            pool=yt_pool,
            is_embedded=is_embedded,
            **kwargs
        )

    @cached_property
    def cooked_soup_edges_path(self):
        return self._get_default_path(self.COOKED_SOUP_EDGES_PATH)

    @cached_property
    def cooked_soup_vertices_properties_path(self):
        return self._get_default_path(self.COOKED_SOUP_VERTICES_PROPERTIES_PATH)

    @cached_property
    def output_dir(self):
        return yt.ypath.ypath_dirname(self.output_edges)

    def get_libs(self):
        return []

    def get_context_data(self, **kwargs):
        return super(Sampler, self).get_context_data(
            soup_edges_table=self.cooked_soup_edges_path,
            vertices_properties_table=self.cooked_soup_vertices_properties_path,
            components_table=self.components_table,
            output_edges=self.output_edges,
            output_properties=self.output_properties,
            **kwargs
        )

    def run(self, **kwargs):
        if not yt.exists(self.output_dir):
            yt.create("map_node", self.output_dir, recursive=True, ignore_existing=True)

        with yt.Transaction() as transaction:  # yql transaction
            super(Sampler, self).run(transaction_id=str(transaction.transaction_id), **kwargs)

            if yt.has_attribute(self.cooked_soup_edges_path, "generate_date"):
                date = yt.get_attribute(self.cooked_soup_edges_path, "generate_date")
                yt.set('{path}/@{attr}'.format(path=self.output_edges, attr=self.GENERATE_DATE_ATTR), date)
            if yt.has_attribute(self.cooked_soup_vertices_properties_path, "generate_date"):
                date = yt.get_attribute(self.cooked_soup_vertices_properties_path, "generate_date")
                yt.set('{path}/@{attr}'.format(path=self.output_properties, attr=self.GENERATE_DATE_ATTR), date)


class RandomSampler(Sampler):
    QUERY_TEMPLATE = "/random_sample_query.sql"
    PERCENT_ATTR = "percent"

    def __init__(self, components_table, percent, output_edges, output_properties, yt_proxy=None, yt_pool="crypta", is_embedded=False, **kwargs):
        super(RandomSampler, self).__init__(
            components_table=components_table,
            output_edges=output_edges,
            output_properties=output_properties,
            yt_proxy=yt_proxy,
            yt_pool=yt_pool,
            is_embedded=is_embedded,
            **kwargs
        )
        self.percent = percent

    def get_context_data(self, **kwargs):
        return super(RandomSampler, self).get_context_data(
            percent=self.percent,
            **kwargs
        )

    def run(self, **kwargs):
        logger.info("Start sampling {perc:.0%} percent of MRCC-components".format(perc=self.percent))

        super(RandomSampler, self).run(**kwargs)

        yt.set('{path}/@{attr}'.format(path=self.output_edges, attr=self.PERCENT_ATTR), self.percent)
        yt.set('{path}/@{attr}'.format(path=self.output_properties, attr=self.PERCENT_ATTR), self.percent)

        logger.info("Finish sampling")


class StaffSampler(Sampler):
    QUERY_TEMPLATE = "/staff_sample_query.sql"
    STAFF_PATH = "//home/crypta/{crypta_env}/graph/staff"

    @cached_property
    def staff_path(self):
        return self._get_default_path(self.STAFF_PATH)

    def get_context_data(self, **kwargs):
        return super(StaffSampler, self).get_context_data(
            staff_table=self.staff_path,
            **kwargs
        )

    def run(self, **kwargs):
        logger.info("Start staff-sampling of MRCC-components")

        super(StaffSampler, self).run(**kwargs)

        logger.info("Finish sampling")
