# -*- coding: utf-8 -*-
from __future__ import print_function
import logging

from argparse import ArgumentParser
from ads.bsyeti.tools.solomon.lib.solomon_api import SolomonAPI

from crypta.graph.solomon.dashboards import (
    brusilov,
    he_resharder,
    herschel,
    menu_utils,
    michurin,
    resharder,
)


def main():  # noqa: C901  # noqa
    dashboards_modules = {
        "brusilov": brusilov,
        "he_resharder": he_resharder,
        "herschel": herschel,
        "michurin": michurin,
        "resharder": resharder,
    }

    logging.basicConfig(
        format=u"%(filename)s[LINE:%(lineno)d]# %(levelname)-8s [%(asctime)s]  %(message)s",
        level=logging.INFO,
    )

    parser = ArgumentParser(description="Generate solomon dashboards for crypta")

    parser.add_argument(
        "-r",
        "--run",
        action="store_false",
        dest="dry_run",
        help="Really run (dry-run - by default)",
    )
    parser.add_argument(
        "-p",
        "--projects",
        nargs="+",
        dest="projects",
        default=list(sorted(dashboards_modules.keys())),
        choices=list(sorted(dashboards_modules.keys())),
        help="Projects to be updated",
    )

    args = parser.parse_args()

    logging.info("Updating dashboards: %s", ",".join(sorted(args.projects)))
    logging.info(
        "Updating menus for: %s",
        ",".join(sorted(name for name, module in dashboards_modules.items())),
    )

    sapi = SolomonAPI()

    top_entry_items = []
    menu_items = menu_utils.top_level_template()
    for name in sorted(dashboards_modules):
        module = dashboards_modules[name]
        if (
            name in args.projects
            and getattr(module, "generate_dashboards", None) is not None
        ):
            for dashboard in module.generate_dashboards():
                assert sapi.validate_entity_dict(dashboard)
                if args.dry_run:
                    success, comment = True, "dry-run"
                else:
                    success, comment = sapi.push_dashboard(dashboard)
                logging.info(
                    "Create or update dashboard %s/%s. success=%s, comment=%s",
                    dashboard["projectId"],
                    dashboard["id"],
                    str(success),
                    comment,
                )

        if getattr(module, "generate_menu_items", None) is not None:
            new_menu_items = module.generate_menu_items()
            for entry, items in new_menu_items.items():
                menu_items[entry].extend(items)
            top_entry_items.append(
                {"title": module.MENU_ITEM_TITLE, "items": new_menu_items}
            )

    if menu_items:
        menu = menu_utils.generate_menu(menu_items, top_entry_items)
        if args.dry_run:
            success, comment = True, "dry-run"
        else:
            success, comment = sapi.push_menu(menu)
        logging.info(
            "Create or update menu for %s. success=%s, comment=%s",
            menu["projectId"],
            str(success),
            comment,
        )


if __name__ == "__main__":
    main()
