# coding: utf-8

import itertools
import logging
import urllib

from ads.bsyeti.tools.solomon.lib.source_urls import gen_dashboard_autogen_line
from ads.bsyeti.tools.solomon.dashboards.lib.generator import DashboardGenerator

from crypta.graph.solomon.dashboards import menu_utils

MTR_ARR, EXT_ARR = menu_utils.MTR_ARR, menu_utils.EXT_ARR

logger = logging.getLogger(__name__)
EVENTS_QYT = "//home/crypta/rtsklejka/qyt/sharded_events"
PROJECT = menu_utils.PROJECT
SERVICE = "Michurin Resharder"
DC_LIST = ("Man", "Sas", "Iva", "Vla", "Myt")
TEXT_RESHARDER_MD = """
### {service} ({{count}} logs)
* code: `crypta/graph/rtsklejka/resharder`
""".format(
    service=SERVICE
)

MENU_ITEM_TITLE = u"Resharder {0}".format(MTR_ARR)

dashboardGenerator = DashboardGenerator(
    project=PROJECT,
    magic_prefix_map={
        "Z": "",
        "H": "l.host={{host}}&",
        "S": "l.servant={{servant}}&",
        "T": "l.src_topic={{src_topic}}&",
        "D": "l.src_dc={{src_dc}}&",
        "C": "l.cs=-&",
    },
)
item, row, make_grid = (
    dashboardGenerator.item,
    dashboardGenerator.row,
    dashboardGenerator.make_grid,
)


def topic_in_new_format(topic):
    return topic.replace("--", "/").replace("@", "/")


def topic_in_old_format(topic):
    parts = topic_in_new_format(topic).split("/")
    assert len(parts) >= 2
    return "{0}--{1}".format("@".join(parts[:-1]), parts[-1])


def get_enabled_topics(testing=False):
    """Return input topics, for filter is enabled"""
    stand = "test" if testing else "prod"
    lb_topics = ["crypta/prod/graph/soup", "crypta/{}/graph/rewind".format(stand)]
    return {0: lb_topics}


def make_suppliers(testing=False):
    stand = "test" if testing else "prod"
    return {
        "crypta/prod/graph/soup": {
            "alias": "man_soup_log",
            "src_topic": "crypta/prod/graph/soup",
            "src_dc": "lbk",
            "output_queue": EVENTS_QYT,
            "row_processing_params": {
                "MessageType": "SOUP",
                "Parser": {
                    "FromProtoSoup": {"TimeStamp": "EventTime", "B64Encoding": True}
                },
                "ShardingAlgorithm": "CryptaId",
            },
            "output_codec": "zstd_6",
        },
        "crypta/{}/graph/rewind".format(stand): {
            "alias": "man_soup_log",
            "src_topic": "crypta/{}/graph/rewind".format(stand),
            "src_dc": "lbk",
            "output_queue": EVENTS_QYT,
            "row_processing_params": {
                "MessageType": "SOUP",
                "Parser": {
                    "FromProtoSoup": {"TimeStamp": "EventTime", "B64Encoding": True}
                },
                "ShardingAlgorithm": "CryptaId",
            },
            "output_codec": "zstd_6",
        },
    }


def lb_message_lag_url_format(src_dc, testing):
    testing = True  # TODO (mskorokhod): change prod/test!
    consumer = "crypta/{}/graph/engine/consumer".format("test" if testing else "prod")

    if src_dc == "lbkx":
        return (
            "/"
            "?cluster=lbkx"
            "&project=kikimr"
            "&partition=-"
            "&service=pqtabletAggregatedCounters"
            "&user_counters=PersQueue"
            "&l.ConsumerPath={consumer}"
            "&TopicPath={{topic_path}}"
            "&sensor=TotalMessageLagByLastRead"
            "&graph=auto"
            "&stack=false"
        ).format(consumer=consumer)
    else:
        return (
            "/"
            "?cluster=lbk"
            "&project=kikimr"
            "&partition=-"
            "&service=pqtabletAggregatedCounters"
            "&user_counters=PersQueue"
            "&ConsumerPath={consumer}"
            "&l.host={dc_list}"
            "&l.OriginDC=*"
            "&TopicPath={{topic_path}}"
            "&l.sensor=TotalMessageLagByLastRead"
            "&graph=auto"
            "&stack=false"
        ).format(consumer=consumer, dc_list="|".join(DC_LIST))


def yt_message_lag_url_format(testing, supplier_type):
    return (
        "/?project={project}"
        "&cluster=push_resharder{srv_sfx}"
        "&service=push_resharder{srv_sfx}"
        "&sensor=lag"
        "&l.supplier_type={supplier_type}"
        "&graph=auto"
        "&overLinesTransform=SUMMARY"
        "&checks=%2Bsum"
        "&l.yt_path={{yt_path}}"
        "&b=1d"
        "&e="
    ).format(
        srv_sfx="_9" if testing else "", supplier_type=supplier_type, project=PROJECT
    )


def get_shads_lag_graphs(testing):
    fixtures = make_suppliers(testing=testing)

    graphs = []
    for topic in get_enabled_topics(testing)[0]:
        fixture = fixtures[topic]
        if "src_yt_directory" in fixture:
            graphs.append(
                item(
                    topic,
                    "",
                    yt_message_lag_url_format(testing, "yt_directory").format(
                        yt_path=fixture["src_yt_directory"]
                    ),
                )
            )
        elif "src_yt_queue" in fixture:
            graphs.append(
                item(
                    topic,
                    "",
                    yt_message_lag_url_format(testing, "yt_queue").format(
                        yt_path=fixture["src_yt_queue"]
                    ),
                )
            )
        else:
            graphs.append(
                item(
                    topic,
                    "",
                    lb_message_lag_url_format(fixture.get("src_dc"), testing).format(
                        topic_path=urllib.parse.quote_plus(topic)
                    ),
                )
            )

    return graphs


def gen_message_lag_dashboard(testing=False):
    graphs = get_shads_lag_graphs(testing)

    return {
        "projectId": PROJECT,
        "id": "michurin_message_lag_{}".format("test" if testing else "prod"),
        "name": "{service} {env} MessageLag".format(
            service=SERVICE, env=("Test" if testing else "Prod")
        ),
        "description": gen_dashboard_autogen_line(),
        "heightMultiplier": 1,
        "rows": (
            sum(
                (
                    # [row(item(markdown=TEXT_RESHARDER_MD.format(count=len(graphs)), colspan=3, rowspan=1))],
                    [row(*row_items) for row_items in make_grid(graphs, columns=1)],
                ),
                [],
            )
        ),
    }


def get_written_dashboard(sensor="bytes"):
    topics = [t.replace("--", "/") for t in get_enabled_topics()[0]]

    url_format = (
        "/"
        "?cluster=lbk"
        "&project=kikimr"
        "&service=pqproxy_writeSession"
        "&l.host={dc_list}"
        "&l.OriginDC=cluster"
        "&l.Topic=%21total"
        "&Producer=%21total"
        "&l.Account=*"
        "&l.TopicPath={topic_path}"
        "&l.sensor={sensor}WrittenOriginal"
        "&graph=auto"
        "&stack=false"
    )

    graphs = [
        item(
            topic,
            "",
            url_format.format(
                topic_path=urllib.parse.quote_plus(topic),
                sensor=sensor.title(),
                dc_list="%7C".join(DC_LIST),
            ),
        )
        for topic in topics
    ]

    return {
        "projectId": PROJECT,
        "id": "{}_written".format(sensor),
        "name": "{service} Written {sensor}".format(
            service=SERVICE, sensor=sensor.title()
        ),
        "description": gen_dashboard_autogen_line(),
        "heightMultiplier": 1,
        "rows": (
            sum(
                (
                    # [row(item(markdown=TEXT_RESHARDER_MD.format(count=len(topics)), colspan=3, rowspan=1))],
                    [row(*row_items) for row_items in make_grid(graphs, columns=1)],
                ),
                [],
            )
        ),
    }


def gen_time_lag_dashboard(dc):
    fixtures = make_suppliers(testing=False)
    topics = [
        (topic_in_old_format(topic), fixtures[topic].get("src_dc", ""))
        for topic in get_enabled_topics()[0]
    ]
    consumer = "crypta@test@graph@engine@consumer"

    lbk_url_format = (
        "/"
        "?cluster=lbk"
        "&project=kikimr"
        "&service=pqproxy_readTimeLag"
        "&l.OriginDC={dc}"
        "&l.sensor=TimeLags"
        "&l.Client={consumer}"
        "&l.Topic={{topic}}"
        "&l.Interval=*"
        "&l.host=cluster"
        "&graph=auto"
        "&l.Account=*"
    ).format(dc=dc, consumer=consumer)

    lbkx_url_format = (
        "/"
        "?cluster=lbkx"
        "&project=kikimr"
        "&service=pqproxy_readTimeLag"
        "&Account=*"
        "&host=cluster"
        "&sensor=TimeLags"
        "&Interval=*"
        "&l.OriginDC=cluster"
        "&Topic={{topic}}"
        "&l.Client={consumer}"
        "&graph=auto"
    ).format(consumer=consumer)

    def url_format(src_dc):
        return lbkx_url_format if src_dc == "lbkx" else lbk_url_format

    graphs = [
        item(topic, "", url_format(src_dc).format(topic=topic))
        for topic, src_dc in topics
    ]

    return {
        "projectId": PROJECT,
        "id": "time_lag_{0}".format(dc.lower()),
        "name": "{service} TimeLag {dc}".format(service=SERVICE, dc=dc.lower()),
        "description": gen_dashboard_autogen_line(),
        "heightMultiplier": 1,
        "rows": (
            sum(
                (
                    # [row(item(markdown=TEXT_RESHARDER_MD.format(count=len(topics)), colspan=1, rowspan=1))],
                    [row(*row_items) for row_items in make_grid(graphs, columns=3)],
                ),
                [],
            )
        ),
    }


def gen_load_dashboard_v3():
    return {
        "id": "resharder_load_v3",
        "projectId": PROJECT,
        "name": "{service} load".format(service=SERVICE),
        "description": gen_dashboard_autogen_line(),
        "heightMultiplier": 2,
        "parameters": [
            {"name": "host", "value": "cluster"},
            {"name": "servant", "value": "resharder"},
            {"name": "src_topic", "value": "all_topics"},
            {"name": "src_dc", "value": "all_dcs"},
            {"name": "cluster", "value": "*"},
            {"name": "service", "value": "*"},
        ],
        "rows": [
            # row(
            #     item("CPU of {{servant}}", "", "H#sensor=cpu_usage_of_{{servant}}"),
            #     item("Memory", "", "HS#place=/&sensor=current_rss"),
            #     item("NetRecv of {{host}}", "", "H#partition=recv&sensor=net_io_bytes&info=net_io"),
            #     item("NetSend of {{host}}", "", "H#partition=sent&sensor=net_io_bytes&info=net_io"),
            #     # item(
            #     #     "Mentor config versions", "",
            #     #     "#host=cluster&mentor_conf_version=*&servant=mentor&hideNoData=true"
            #     # ),
            # ),
            row(
                item(
                    "Young consuming systems",
                    "",
                    "HSTDC#sensor=bigrt.cs.balancer_young|bigrt.cs.system_young&min=0",
                ),
                item(
                    "Young mediators",
                    "",
                    "HSTDC#sensor=bigrt.cs.mediator_iteration_young|bigrt.cs.mediator_young&shard=-&min=0",
                ),
                item(
                    "Balancer current shards",
                    "",
                    "HSTDC#sensor=bigrt.cs.current_shards",
                ),
                item(
                    "Commits",
                    "",
                    "HSTDC#sensor=commits_*&place=stateless_processor&shard=-&stack=false&min=0",
                ),
            ),
            row(
                item(
                    "Input messages",
                    "",
                    "HSTDC#sensor=bigrt.cs.messages_total&shard=-&supplier=*",
                ),  # todo supplier all
                item(
                    "Processed bytes",
                    "",
                    "HSTDC#place=stateless_processor&sensor=processed_bytes&shard=-",
                ),
                item(
                    "Rows count input & output & skipped",
                    "",
                    "HSTDC#place=reshard_batches&sensor=rows_count_*&shard=-&stack=false&min=0",
                ),
                item(
                    "Commit timings",
                    "",
                    "HSTDC#sensor=*&place=stateless_processor&subplace=commit_detailed",
                ),
            ),
            row(
                item(
                    "Inflight",
                    "",
                    "HSTDC#sensor=bigrt.inflight_limiter.bytes&stack=false&min=0",
                ),
                item(
                    "Sharding times",
                    "",
                    "HSTDC#place=reshard_batches&subplace=reshard_detailed&sensor=*_total_time&shard=-&stack=true",
                ),
                item(
                    "Processing times",
                    "",
                    "HSTDC#place=reshard_batches&subplace=-&sensor=*_total_time&shard=-&"
                    "stack=true&norm=true&min=0&hideNoData=true",
                ),
                item(
                    "Pure commit timings distribution",
                    "",
                    "HSTDC#&place=stateless_processor&l.subplace=-&l.sensor=commit_time&"
                    "norm=true&stack=false&l.shard=-",
                ),
            ),
            row(
                item(
                    "Average wait in heavy threads queue",
                    "(common for all src_dc and topics)",
                    "H#sensor=resharder_heavy_average_wait_time_hist&stack=false&norm=true",
                ),
                item(
                    "Wait for data & inflight",
                    "",
                    "HSTDC#place=stateless_processor&sensor=wait_for_*_total_time",
                ),
                item(
                    "Trottle time",
                    "",
                    "HSTDC#place=reshard_batches&sensor=*trottle_total_time&shard=-&stack=false&min=0",
                ),
                item(
                    "From epoch begin to epoch commited",
                    "",
                    "HSTDC#&place=stateless_processor&l.subplace=-&"
                    "l.sensor=epoch_from_begin_to_commited&norm=true&stack=false&l.shard=-",
                ),
            ),
            row(
                item(
                    "Versions",
                    "",
                    "#host=*&checks=-cluster;{skip}&sensor=revision&hideNoData=true"
                    "&stack=false&numberFormat=0|".format(skip=";".join(DC_LIST)),
                ),
                item(
                    "Build Time",
                    "",
                    "#host=*&checks=-cluster;{skip}&sensor=build_time&hideNoData=true"
                    "&stack=false&numberFormat=0|".format(skip=";".join(DC_LIST)),
                ),
            ),
            row(
                item(
                    "Memory Load",
                    "",
                    "H#sensor=yt.resource_tracker.memory_usage.rss&stack=true&l.src_dc=-&min=0",
                ),
                item(
                    "CPU Usage",
                    "",
                    "H#sensor=yt.resource_tracker.*cpu&l.thread=-&stack=true&l.src_dc=-&min=0",
                ),
            ),
        ],
    }


def gen_identification_dashboard():
    return {
        "id": "resharder_identification",
        "projectId": PROJECT,
        "name": "{service} idfy".format(service=SERVICE),
        "description": gen_dashboard_autogen_line(),
        "heightMultiplier": 2,
        "parameters": [
            {"name": "host", "value": "cluster"},
            {"name": "servant", "value": "resharder"},
            {"name": "src_topic", "value": "all_topics"},
            {"name": "src_dc", "value": "all_dcs"},
            {"name": "cluster", "value": "*"},
            {"name": "service", "value": "*"},
        ],
        "rows": [
            row(
                item(
                    "Requests",
                    "",
                    "HSTD#sensor=requests_*&place=identification&shard=-&stack=false&min=0",
                ),
                item(
                    "Rows",
                    "",
                    "HSTD#sensor=rows_*&place=identification&shard=-&stack=false&min=0",
                ),
            ),
            row(
                item(
                    "Interesting",
                    "",
                    "HSTD#sensor=interesting_identifiers_*&place=identification&shard=-&stack=false&min=0",
                ),
                item(
                    "Deduplicated",
                    "",
                    "HSTD#sensor=deduplicated_identifiers_*&place=identification&shard=-&stack=false&min=0",
                ),
            ),
            row(
                item(
                    "Identifiers",
                    "",
                    "HSTD#sensor=identifiers_*&place=identification&shard=-&stack=false&min=0",
                ),
                item(
                    "Resolved",
                    "",
                    "HSTD#sensor=id_*_crypta_id&place=identification&shard=-&stack=false&min=0",
                ),
            ),
        ],
    }


def gen_load_dashboard_info():
    summary = ""
    return {
        "id": "resharder_info",
        "projectId": PROJECT,
        "name": "{service} info".format(service=SERVICE),
        "description": gen_dashboard_autogen_line(),
        "heightMultiplier": 2,
        "parameters": [
            {"name": "host", "value": "cluster"},
            {"name": "src_topic", "value": "all_topics"},
            {"name": "src_dc", "value": "all_dcs"},
            {"name": "cluster", "value": "*"},
            {"name": "service", "value": "*"},
        ],
        "rows": [
            row(
                item(
                    "Revision",
                    "",
                    "ZC#&checks=-cluster;{skip}&sensor=revision&stack=false&numberFormat=0|".format(
                        skip=";".join(DC_LIST)
                    ),
                ),
                item(
                    "DeployLogs",
                    "",
                    "Z#project=kikimr&cluster=ydb_deploy_sas&service=qloud_deploy_ng_pusher"
                    "&host=cluster&sensor=BytesWritten&log=crypta/docker/crypta-rtsklejka-resharder",
                ),
            ),
            row(
                item(
                    "Messages",
                    "",
                    "HC#sensor=messages_count_*&stack=false{}".format(summary),
                ),
                item(
                    "Rows", "", "HC#sensor=rows_count_*&stack=false{}".format(summary)
                ),
            ),
            row(
                item(
                    "Identification", "", "HC#sensor=id_*_crypta_id{}".format(summary)
                ),
                item(
                    "Total time", "", "HC#sensor=*_total_time&subplace=reshard_detailed"
                ),
            ),
            row(
                item(
                    "Wait queue",
                    "",
                    "HC#sensor=wait_queue_total_time{}".format(summary),
                ),
                item(
                    "Wait data",
                    "",
                    "HC#sensor=wait_for_data_total_time{}".format(summary),
                ),
            ),
        ],
    }


def generate_dashboard_info(stand="prod"):
    stand_name = "resharder-{stand}".format(stand=stand)
    info_url_template = (
        "/"
        "?project={project}"
        "&cluster={stand_name}"
        "&service={stand_name}"
        "&dashboard=resharder_info"
        "&l.host=cluster"
        "&l.servant=resharder"
        "&src_topic=all_topics"
        "&src_dc=all_dcs"
        "&b=12h"
    )
    load_url_template = (
        "/"
        "?project={project}"
        "&cluster={stand_name}"
        "&service={stand_name}"
        "&dashboard=resharder_load_v3"
        "&l.host=cluster"
        "&l.servant=resharder"
        "&src_topic=*"
        "&src_dc=*"
        "&b=1h30m"
    )
    idfy_url_template = (
        "/"
        "?project={project}"
        "&cluster={stand_name}"
        "&service={stand_name}"
        "&dashboard=resharder_identification"
        "&l.host=cluster"
        "&l.servant=resharder"
        "&src_topic=*"
        "&src_dc=*"
        "&b=1h30m"
    )
    hosts_url_template = (
        "/"
        "?project={project}"
        "&cluster={stand_name}"
        "&service={stand_name}"
        "&dashboard=resharder_hosts"
        "&l.host=cluster"
        "&l.servant=resharder"
        "&b=1h30m"
        "&l.src_topic=all_topics"
    )

    return sum(
        (
            [
                {
                    "name": "{service} {stand}".format(service=SERVICE, stand=stand),
                    "group": "Info",
                    "important": True,
                    "url": info_url_template.format(
                        project=PROJECT, stand_name=stand_name
                    ),
                }
            ],
            [
                {
                    "name": "{service} {stand} load".format(
                        service=SERVICE, stand=stand
                    ),
                    "group": "Load",
                    "important": True,
                    "url": load_url_template.format(
                        project=PROJECT, stand_name=stand_name
                    ),
                }
            ],
            [
                {
                    "name": "{service} {stand}".format(service=SERVICE, stand=stand),
                    "group": "Identification",
                    "important": True,
                    "url": idfy_url_template.format(
                        project=PROJECT, stand_name=stand_name
                    ),
                }
            ],
            [
                {
                    "name": "{service} {stand} messages written".format(
                        service=SERVICE, stand=stand
                    ),
                    "group": "Written",
                    "important": True,
                    "url": "/?project={project}&dashboard=messages_written".format(
                        project=PROJECT
                    ),
                },
                {
                    "name": "{service} {stand} bytes written".format(
                        service=SERVICE, stand=stand
                    ),
                    "group": "Written",
                    "important": True,
                    "url": "/?project={project}&dashboard=bytes_written".format(
                        project=PROJECT
                    ),
                },
            ],
            [
                {
                    "name": "{service} {stand} message lag".format(
                        service=SERVICE, stand=stand
                    ),
                    "group": "Lag",
                    "important": True,
                    "url": "/?project={project}&dashboard=michurin_message_lag_{stand}".format(
                        project=PROJECT, stand=stand
                    ),
                }
            ],
            [
                {
                    "name": "TimeLag {}".format(host.capitalize()),
                    "group": "Lag",
                    "important": True,
                    "url": "/?project={project}&dashboard=time_lag_{host}".format(
                        project=PROJECT, host=host.lower()
                    ),
                }
                for host in itertools.chain(["cluster"], DC_LIST)
            ],
            [
                {
                    # not enabled now!
                    "name": "{service} {stand} hosts".format(
                        service=SERVICE, stand=stand
                    ),
                    "group": "Hosts",
                    "important": True,
                    "url": hosts_url_template.format(
                        project=PROJECT, stand_name=stand_name
                    ),
                }
            ],
        ),
        [],
    )


def _generate_resharder_menu_items():
    def base_generate_menu_items(line):
        queue_url_template = (
            "/"
            "?project={project}"
            "&dashboard=queue_lags_and_rate"
            "&cluster={yt_cluster}"
            "&l.consumer={consumer}"
            "&l.queue={queue}"
            "&b=6h"
        )
        return {
            "title": line["title"],
            "url": queue_url_template.format(project=PROJECT, **line),
        }

    return map(
        base_generate_menu_items,
        [
            dict(
                title="Resharder output",
                consumer="resharder",
                yt_cluster="zeno",
                queue="//home/crypta/rtsklejka/qyt/sharded_events",
            )
        ],
    )


def generate_dashboards():
    return sum(
        (
            [gen_load_dashboard_v3()],
            [gen_load_dashboard_info()],
            [gen_identification_dashboard()],
            # [gen_hosts_dashboard()],
            [gen_message_lag_dashboard(testing=testing) for testing in [False, True]],
            [get_written_dashboard(sensor) for sensor in ["bytes", "messages"]],
            [
                gen_time_lag_dashboard(dc)
                for dc in itertools.chain(["cluster"], DC_LIST)
            ],
        ),
        [],
    )


def lb_resharder_consumers(env):
    return [
        {
            "title": u"{service} {env} {ext}".format(
                service=SERVICE, env=env.upper(), ext=EXT_ARR
            ),
            "url": "https://lb.yandex-team.ru/main/consumer/crypta/{0}/graph/engine/consumer/topics".format(
                env
            ),
        },
        {
            "title": u"β {service} {env} {ext}".format(
                service=SERVICE, env=env.upper(), ext=EXT_ARR
            ),
            "url": (
                "https://beta.logbroker.yandex-team.ru"
                "/logbroker/accounts/crypta/{0}/graph/engine/consumer"
                "?group=all"
                "&page=statistics"
                "&type=consumer"
                "&tab=topics"
                "&shownTopics=all%20original"
                "&topicCluster=all%20original"
                "&consumerView=Grouped%20view"
            ).format(env),
        },
    ]


def generate_menu_items():
    items = menu_utils.top_level_template()

    for info in generate_dashboard_info("prod") + generate_dashboard_info("test"):
        if info["group"] in items:
            items[info["group"]].append({"title": info["name"], "url": info["url"]})

    for env in ["test", "prod"]:
        items["LB Consumers"].extend(lb_resharder_consumers(env))

    items["Deploy"].extend(
        [
            {
                "title": u"{service} {stand} {ext}".format(
                    service=SERVICE, stand="prod", ext=EXT_ARR
                ),
                "url": "https://deploy.yandex-team.ru/stage/crypta-rtsklejka-resharder-prod",
            },
            {
                "title": u"{service} {stand} {ext}".format(
                    service=SERVICE, stand="test", ext=EXT_ARR
                ),
                "url": "https://deploy.yandex-team.ru/stage/crypta-rtsklejka-resharder",
            },
        ]
    )
    items["YT"].extend(
        [
            {
                "title": "YT Artemis",
                "url": "/?project=yt&dashboard=yt_artemis_bundle_v2&service=yt_node_tablet_profiling&"
                "cluster=hahn&tablet_cell_bundle=crypta-graph&b=1d&e=",
            },
            {
                "title": "YT RPC Proxy",
                "url": "/?project=yt&cluster=hahn&service=rpc_proxy_rpc&l.method=LookupRows"
                "&l.sensor=yt.rpc.server.*request_count.rate&l.user=robot-crypta"
                "&l.host=Aggr&graph=auto&stack=false",
            },
        ]
    )
    return items
