#include "soup_config.h"

#include <library/cpp/resource/resource.h>

#include <google/protobuf/message.h>
#include <google/protobuf/text_format.h>
#include <google/protobuf/descriptor.pb.h>

#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/generic/singleton.h>

template <>
struct TLess<NCrypta::NSoup::TEdgeType> {
    bool operator()(const NCrypta::NSoup::TEdgeType& lhs, const NCrypta::NSoup::TEdgeType& rhs) const {
        const auto l = std::make_tuple(lhs.GetId1Type(), lhs.GetId2Type(), lhs.GetSourceType(), lhs.GetLogSource());
        const auto r = std::make_tuple(rhs.GetId1Type(), rhs.GetId2Type(), rhs.GetSourceType(), rhs.GetLogSource());
        return l < r;
    }
};

namespace NCrypta {
    namespace NSoup {
        namespace {
            using google::protobuf::Descriptor;
            using google::protobuf::EnumDescriptor;
            using google::protobuf::EnumValueDescriptor;
            using google::protobuf::EnumValueOptions;
            using google::protobuf::FieldDescriptor;
            using google::protobuf::Reflection;

            using NCrypta::NIdentifiersProto::NIdType::EIdType_Name;
            using NSourceType::ESourceType_Name;
            using NLogSource::ELogSourceType_Name;
            using NCrypta::NIdentifiersProto::NIdType::EIdType_descriptor;
            using NSourceType::ESourceType_descriptor;
            using NLogSource::ELogSourceType_descriptor;

            template <typename E, typename M>
            class TProtobufEnum {
            public:
                THashMap<E, M> ByType;
                THashMap<TString, M> ByName;

                TProtobufEnum(const EnumDescriptor* enumDesc) {
                    for (int i = 0; i < enumDesc->value_count(); i++) {

                        M val;

                        const Descriptor* msgDesc = M::descriptor();
                        const Reflection* valRefl = val.GetReflection();

                        const EnumValueDescriptor* enumValDesc = enumDesc->value(i);
                        E e = static_cast<E>(enumValDesc->number());
                        val.SetType(e);

                        const EnumValueOptions& opts = enumValDesc->options();

                        // I don't know how to get the name of an extension in C++ :(
                        TString soupName = opts.GetExtension(NCrypta::NSoup::Name);
                        if (soupName) {
                            valRefl->SetString(&val, msgDesc->FindFieldByName("Name"), soupName);
                            ByName.insert({soupName, val});
                        }

                        TString descr = opts.GetExtension(Description);
                        if (descr) {
                            valRefl->SetString(&val, msgDesc->FindFieldByName("Description"), descr);
                        }

                        TString logPath = opts.GetExtension(LogPath);
                        if (logPath) {
                            valRefl->SetString(&val, msgDesc->FindFieldByName("LogPath"), logPath);
                        }

                        ERepresentedObjectType repObj = opts.GetExtension(NCrypta::NIdentifiersProto::RepObj);
                        if (repObj) {
                            valRefl->SetEnumValue(&val, msgDesc->FindFieldByName("RepObj"), repObj);
                        }

                        TString idTypeName = opts.GetExtension(NCrypta::NIdentifiersProto::Name);
                        if (idTypeName) {
                            valRefl->SetString(&val, msgDesc->FindFieldByName("Name"), idTypeName);
                            ByName.insert({idTypeName, val});
                        }

                        if (soupName && idTypeName) {
                            ythrow yexception() << "Name extensions clash";
                        }

                        ByType.insert({e, val});
                    }
                }
            };

            class TSoupConfig {
            public:
                TMap<TEdgeType, TEdgeProps> EdgeProps;
                TMap<TEdgeType, TEdgeUsage> EdgeUsage;
                TVector<TEdgeType> Edges;
                TVector<TEdgeRecord> Records;
                TProtobufEnum<EIdType, TIdType> idTypes;
                TProtobufEnum<ESourceType, TSourceType> sourceTypes;
                TProtobufEnum<ELogSourceType, TLogSourceType> logSources;

                TSoupConfig()
                    : idTypes(EIdType_descriptor())
                    , sourceTypes(ESourceType_descriptor())
                    , logSources(ELogSourceType_descriptor())
                {
                    TEdges edges;
                    TString protoTxt = NResource::Find("/edge_types.pb.txt");
                    google::protobuf::TextFormat::ParseFromString(protoTxt, &edges);

                    for (const TEdgeRecord& er : edges.GetEdges()) {
                        Records.push_back(er);
                        Edges.push_back(er.GetType());
                        EdgeProps.insert({er.GetType(), er.GetProps()});
                        EdgeUsage.insert({er.GetType(), er.GetUsage()});

                        // work with synthetic edges (such as bochka)
                        // todo: make another synthetic edges (such as rtmr)
                        if (er.GetUsage().GetBochka()) {
                            TEdgeRecord ephemeral{er};
                            ephemeral.MutableType()->SetLogSource(NLogSource::BOCHKA);

                            Records.push_back(ephemeral);
                            Edges.push_back(ephemeral.GetType());
                            EdgeProps.insert({ephemeral.GetType(), ephemeral.GetProps()});
                            EdgeUsage.insert({ephemeral.GetType(), ephemeral.GetUsage()});
                        }
                    }
                }
            };
        }

        const TVector<TEdgeType>& EdgeTypes() {
            return Singleton<TSoupConfig>()->Edges;
        }

        const TVector<TEdgeRecord>& EdgeRecords() {
            return Singleton<TSoupConfig>()->Records;
        }

        const TEdgeType& EdgeType(const EIdType id1, const EIdType id2, const ESourceType source, const ELogSourceType log) {
            TEdgeType et;
            et.SetId1Type(id1);
            et.SetId2Type(id2);
            et.SetSourceType(source);
            et.SetLogSource(log);

            const auto& edgeProps = Singleton<TSoupConfig>()->EdgeProps;
            const auto it = edgeProps.find(et);
            if (it != edgeProps.end()) {
                return it->first;
            } else {
                et.SetId1Type(id2);
                et.SetId2Type(id1);
                const auto itRev = edgeProps.find(et);
                if (itRev != edgeProps.end()) {
                    return itRev->first;
                }
            }

            ythrow yexception() << "EdgeType (" << EIdType_Name(id1) << ", " << EIdType_Name(id2) << ", "
                                                << ESourceType_Name(source) << ", " << ELogSourceType_Name(log) << ") doesn't exist";
        }

        const TEdgeType& EdgeType(const TStringBuf id1, const TStringBuf id2, const TStringBuf source, const TStringBuf log) {
            auto id1Type = IdType(id1);
            auto id2Type = IdType(id2);
            auto sourceType = SourceType(source);
            auto logSource = LogSourceType(log);

            return EdgeType(id1Type.GetType(), id2Type.GetType(), sourceType.GetType(), logSource.GetType());
        }

        const TEdgeProps& EdgeProps(const TEdgeType& edgeType) {
            return Singleton<TSoupConfig>()->EdgeProps.at(edgeType);
        }

        const TEdgeUsage& EdgeUsage(const TEdgeType& edgeType) {
            return Singleton<TSoupConfig>()->EdgeUsage.at(edgeType);
        }

        const TIdType& IdType(const EIdType type) {
            return Singleton<TSoupConfig>()->idTypes.ByType.at(type);
        }

        const TIdType& IdType(const TStringBuf name) {
            return Singleton<TSoupConfig>()->idTypes.ByName.at(name);
        }

        const TSourceType& SourceType(const ESourceType type) {
            return Singleton<TSoupConfig>()->sourceTypes.ByType.at(type);
        }

        const TSourceType& SourceType(const TStringBuf name) {
            return Singleton<TSoupConfig>()->sourceTypes.ByName.at(name);
        }

        const TLogSourceType& LogSourceType(const ELogSourceType type) {
            return Singleton<TSoupConfig>()->logSources.ByType.at(type);
        }

        const TLogSourceType& LogSourceType(const TStringBuf name) {
            return Singleton<TSoupConfig>()->logSources.ByName.at(name);
        }

    }
}
