#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/gtest.h>

#include <crypta/graph/soup/config/cpp/soup_config.h>

using namespace NCrypta::NSoup;

namespace NIdType = NCrypta::NIdentifiersProto::NIdType;
namespace NRepObj = NCrypta::NIdentifiersProto::NRepObj;

TEST(SoupConfig, Deserialize) {
    EXPECT_TRUE(EdgeTypes().size() > 0);
}

TEST(SoupConfig, Props) {
    TEdgeType et;
    et.SetId1Type(NIdType::PHONE);
    et.SetId2Type(NIdType::PHONE_MD5);
    et.SetSourceType(NSourceType::MD5_HASH);
    et.SetLogSource(NLogSource::SOUP_PREPROCESSING);

    EXPECT_TRUE(EdgeProps(et).GetEdgeStrength() == TEdgeProps::ARTIFICIAL);
    EXPECT_TRUE(EdgeProps(et).GetActivityType() == TEdgeProps::NONE);
    EXPECT_TRUE(EdgeProps(et).GetDeviceBounds() == TEdgeProps::CROSSDEVICE);

    et.SetId1Type(NIdType::GAID);
    et.SetId2Type(NIdType::MM_DEVICE_ID);
    et.SetSourceType(NSourceType::APP_METRICA);
    et.SetLogSource(NLogSource::METRIKA_MOBILE_LOG);

    EXPECT_TRUE(EdgeProps(et).GetEdgeStrength() == TEdgeProps::TRUSTED);
    EXPECT_TRUE(EdgeProps(et).GetActivityType() == TEdgeProps::DAILY);
    EXPECT_TRUE(EdgeProps(et).GetDeviceBounds() == TEdgeProps::INDEVICE);
}

TEST(SoupConfig, Usage) {
    TEdgeType et;
    et.SetId1Type(NIdType::YANDEXUID);
    et.SetId2Type(NIdType::UUID);
    et.SetSourceType(NSourceType::SEARCH_APP_MOB_REPORT);
    et.SetLogSource(NLogSource::MOBILE_REPORT_LOG);

    EXPECT_FALSE(EdgeUsage(et).GetSoupUpdate());
    EXPECT_TRUE(EdgeUsage(et).GetHumanMatching());
    EXPECT_TRUE(EdgeUsage(et).GetYandexSafe());
    EXPECT_TRUE(EdgeUsage(et).GetSoupyIndevice());
}

TEST(SoupConfig, Synthetic) {
    TEdgeType et;
    et.SetId1Type(NIdType::YANDEXUID);
    et.SetId2Type(NIdType::DISTR_UI);
    et.SetSourceType(NSourceType::YASOFT);
    et.SetLogSource(NLogSource::BOCHKA);  // real source is BAR_NAVIG_LOG

    EXPECT_TRUE(EdgeProps(et).GetDeviceBounds() == TEdgeProps::INDEVICE);
    EXPECT_TRUE(EdgeUsage(et).GetBochka());
}

TEST(SoupConfig, IdType) {
    EXPECT_EQ(IdType(NIdType::YANDEXUID).GetName(), "yandexuid");
    EXPECT_TRUE(IdType(NIdType::YANDEXUID).GetRepObj() == NRepObj::BROWSER);
    EXPECT_TRUE(IdType("puid").GetType() == NIdType::PUID);
    EXPECT_TRUE(IdType("puid").GetRepObj() == NRepObj::PERSON);
}

TEST(SoupConfig, SourceType) {
    EXPECT_EQ(SourceType(NSourceType::SOCIAL_AUTH).GetName(), "social-auth");
    EXPECT_EQ(SourceType(NSourceType::SOCIAL_AUTH).GetDescription(), "Yandex social auth");
    EXPECT_TRUE(SourceType("app-url-redir").GetType() == NSourceType::APP_URL_REDIR);
}

TEST(SoupConfig, LogSource) {
    EXPECT_EQ(LogSourceType(NLogSource::WATCH_LOG).GetName(), "wl");
    EXPECT_EQ(LogSourceType(NLogSource::WATCH_LOG).GetLogPath(), "//logs/bs-watch-log/1d");
    EXPECT_TRUE(LogSourceType("access").GetType() == NLogSource::ACCESS_LOG);
}

TEST(SoupConfig, EdgeByType) {
    const TEdgeType& et1 = EdgeType(NIdType::IDFA, NIdType::MM_DEVICE_ID, NSourceType::APP_METRICA, NLogSource::METRIKA_MOBILE_LOG);
    const TEdgeType& et2 = EdgeType(NIdType::MM_DEVICE_ID, NIdType::IDFA, NSourceType::APP_METRICA, NLogSource::METRIKA_MOBILE_LOG);
    EXPECT_TRUE(et1.GetId1Type() == et2.GetId1Type());
    EXPECT_TRUE(et1.GetId2Type() == et2.GetId2Type());
}

TEST(SoupConfig, EdgeByName) {
    const TEdgeType& et1 = EdgeType("yandexuid", "uuid", "app-url-redir", "wl");
    const TEdgeType& et2 = EdgeType("uuid", "yandexuid", "app-url-redir", "wl");
    EXPECT_TRUE(et1.GetId1Type() == et2.GetId1Type());
    EXPECT_TRUE(et1.GetId2Type() == et2.GetId2Type());
}

TEST(SoupConfig, IndeviceByName) {
    const TEdgeType& et = EdgeType("yandexuid", "uuid", "app-url-redir", "wl");
    EXPECT_TRUE(EdgeProps(et).GetDeviceBounds() == TEdgeProps::INDEVICE);
}
