package ru.yandex.crypta.graph.soup.config;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.protobuf.Descriptors;
import com.google.protobuf.GeneratedMessage;
import com.google.protobuf.Message;
import com.google.protobuf.ProtocolMessageEnum;

/**
 * Helper class that provides easier access to enum extension values
 * (by converting it into a collection of ordinary protobuf messages).
 */
public class ProtobufEnum<E extends Enum<E> & ProtocolMessageEnum, M extends Message> {

    private Map<E, M> byType;
    private Map<String, M> byName;
    private Map<String, E> nameToTypeMap;
    private Map<E, String> typeToNameMap;

    public ProtobufEnum(Class<E> enumClass,
                        Descriptors.EnumDescriptor enumDesc,
                        Descriptors.Descriptor msgDesc,
                        Message.Builder builder,
                        Collection<GeneratedMessage.GeneratedExtension> exts) {
        byType = new HashMap<>();
        byName = new HashMap<>();
        nameToTypeMap = new HashMap<>();
        typeToNameMap = new HashMap<>();
        Descriptors.FieldDescriptor typeField = msgDesc.findFieldByName("Type");
        Descriptors.FieldDescriptor nameField = msgDesc.findFieldByName("Name");
        for (Descriptors.EnumValueDescriptor enumVal : enumDesc.getValues()) {
            builder.clear();
            builder.setField(typeField, enumVal);

            for (GeneratedMessage.GeneratedExtension ext : exts) {
                Descriptors.FieldDescriptor extField = ext.getDescriptor();
                Descriptors.FieldDescriptor msgField = msgDesc.findFieldByName(extField.getName());

                Object extVal = enumVal.getOptions().getExtension(ext);
                extVal = convertEnumsToEnumDescriptors(extVal);

                builder.setField(msgField, extVal);
            }

            M val = (M) builder.build();

            E type = Enum.valueOf(enumClass, enumVal.getName());
            String name = val.getField(nameField).toString();

            byType.put(type, val);
            byName.put(name, val);
            nameToTypeMap.put(name, type);
            typeToNameMap.put(type, name);
        }
    }

    private Object convertEnumsToEnumDescriptors(Object extVal) {
        if (extVal.getClass().isEnum()) {
            // single value
            extVal = ((ProtocolMessageEnum) extVal).getValueDescriptor();
        } else if (extVal instanceof Collection) {
            // repeated value
            Collection<?> repeated = (Collection<?>) extVal;
            extVal = repeated.stream().map(el -> {
                if (el.getClass().isEnum()) {
                    return ((ProtocolMessageEnum) el).getValueDescriptor();
                } else {
                    return el;
                }
            }).collect(Collectors.toList());
        }
        return extVal;
    }

    public M getByType(E type) {
        return byType.get(type);
    }

    public M getByName(String name) {
        return byName.get(name);
    }

    public String typeToName(E type) {
        return typeToNameMap.get(type);
    }

    public E nameToType(String name) {
        return nameToTypeMap.get(name);
    }

    public Collection<M> getAll() {
        return byType.values();
    }

    public Collection<E> getAllEnums() {
        return byType.keySet();
    }
}
