package ru.yandex.crypta.graph.soup.config;

import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import com.google.common.io.Resources;
import com.google.protobuf.TextFormat;

import ru.yandex.crypta.graph.soup.config.proto.ELogSourceType;
import ru.yandex.crypta.graph.soup.config.proto.ESourceType;
import ru.yandex.crypta.graph.soup.config.proto.SoupEnumExtensions;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeProps;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeRecord;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeType;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeUsage;
import ru.yandex.crypta.graph.soup.config.proto.TEdges;
import ru.yandex.crypta.graph.soup.config.proto.TLogSourceType;
import ru.yandex.crypta.graph.soup.config.proto.TSourceType;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.IdentifiersEnumExtensions;
import ru.yandex.crypta.lib.proto.identifiers.TIdType;

import static ru.yandex.crypta.lib.proto.identifiers.EIdType.GAID;
import static ru.yandex.crypta.lib.proto.identifiers.EIdType.IDFA;
import static ru.yandex.crypta.lib.proto.identifiers.EIdType.OLD_DEVICE_ID;

public class Soup {
    private Map<TEdgeType, TEdgeRecord> edgeTypeConfigs;

    private ProtobufEnum<EIdType, TIdType> idTypes;
    private ProtobufEnum<ELogSourceType, TLogSourceType> logSources;
    private ProtobufEnum<ESourceType, TSourceType> sourceTypes;

    public static final Soup CONFIG;

    static {
        CONFIG = new Soup();
    }

    private Soup() {
        try {
            String pb = Resources.toString(Resources.getResource("edge_types.pb.txt"), StandardCharsets.UTF_8);
            TEdges.Builder bld = TEdges.newBuilder();
            TextFormat.merge(pb, bld);
            TEdges protoEdges = bld.build();

            edgeTypeConfigs = new HashMap<>();
            for (TEdgeRecord r : protoEdges.getEdgesList()) {
                edgeTypeConfigs.put(r.getType(), r);
            }

            idTypes = new ProtobufEnum<>(EIdType.class,
                                         EIdType.getDescriptor(),
                                         TIdType.getDescriptor(),
                                         TIdType.newBuilder(),
                                         ImmutableList.of(
                                            IdentifiersEnumExtensions.name,
                                            IdentifiersEnumExtensions.repObj));

            logSources = new ProtobufEnum<>(ELogSourceType.class,
                                            ELogSourceType.getDescriptor(),
                                            TLogSourceType.getDescriptor(),
                                            TLogSourceType.newBuilder(),
                                            ImmutableList.of(
                                                SoupEnumExtensions.name,
                                                SoupEnumExtensions.logPath));

            sourceTypes = new ProtobufEnum<>(ESourceType.class,
                                             ESourceType.getDescriptor(),
                                             TSourceType.getDescriptor(),
                                             TSourceType.newBuilder(),
                                             ImmutableList.of(
                                                 SoupEnumExtensions.name,
                                                 SoupEnumExtensions.description));
        } catch (Exception e) {
            throw new RuntimeException("Can't initialize soup config!", e);
        }
    }

    public Collection<TEdgeType> getEdgeTypes() { return edgeTypeConfigs.keySet(); }

    /**
     * @return TEdgeType from proto config
     * @throws NoSuchEdgeException if not found
     */
    public TEdgeType getEdgeType(String id1TypeName, String id2TypeName, String sourceTypeName, String logSourceName) {
        TEdgeType edgeType = tryGetEdgeType(id1TypeName, id2TypeName, sourceTypeName, logSourceName);
        if (edgeType == null) {
            throw new NoSuchEdgeException(id1TypeName, id2TypeName, sourceTypeName, logSourceName);
        } else {
            return edgeType;
        }

    }

    /**
     * @return TEdgeType from proto config or null
     */
    public TEdgeType tryGetEdgeType(String id1TypeName, String id2TypeName, String sourceTypeName,
                                    String logSourceName) {
        TIdType id1Type = getIdType(id1TypeName);
        TIdType id2Type = getIdType(id2TypeName);
        TSourceType sourceType = getSourceType(sourceTypeName);
        TLogSourceType logSource = getLogSource(logSourceName);

        if (id1Type == null || id2Type == null || sourceType == null || logSource == null) {
            return null;
        }

        TEdgeType result = TEdgeType.newBuilder()
                .setId1Type(id1Type.getType())
                .setId2Type(id2Type.getType())
                .setSourceType(sourceType.getType())
                .setLogSource(logSource.getType())
                .build();

        if (edgeTypeConfigs.containsKey(result)) {
            return result;
        } else {
            return null;
        }
    }

    public ProtobufEnum<EIdType, TIdType> getIdTypes() {
        return idTypes;
    }

    public ProtobufEnum<ELogSourceType, TLogSourceType> getLogSources() {
        return logSources;
    }

    public ProtobufEnum<ESourceType, TSourceType> getSourceTypes() {
        return sourceTypes;
    }

    public TEdgeRecord getEdgeConfig(TEdgeType edgeType) { return edgeTypeConfigs.get(edgeType); }
    public TEdgeProps getEdgeProps(TEdgeType edgeType) { return edgeTypeConfigs.get(edgeType).getProps(); }
    public TEdgeUsage getEdgeUsage(TEdgeType edgeType) { return edgeTypeConfigs.get(edgeType).getUsage(); }

    public String getEdgeName(TEdgeType edgeType) {
        return String.join("_",
                name(edgeType.getId1Type()),
                name(edgeType.getId2Type()),
                name(edgeType.getSourceType()),
                name(edgeType.getLogSource())
        );
    }

    public TIdType getIdType(EIdType type) { return idTypes.getByType(type); }
    public TIdType getIdType(String name) { return idTypes.getByName(name); }
    public String name(EIdType type) { return idTypes.typeToName(type); }

    public TLogSourceType getLogSource(ELogSourceType type) { return logSources.getByType(type); }
    public TLogSourceType getLogSource(String name) { return logSources.getByName(name); }
    public String name(ELogSourceType type) { return logSources.typeToName(type); }

    public TSourceType getSourceType(ESourceType type) { return sourceTypes.getByType(type); }
    public TSourceType getSourceType(String name) { return sourceTypes.getByName(name); }
    public String name(ESourceType type) { return sourceTypes.typeToName(type); }

    public static final List<EIdType> MAIN_DEVICE_IDS = List.of(IDFA, GAID, OLD_DEVICE_ID);
    public boolean isDeviceIdMainId(EIdType idType) {
        return MAIN_DEVICE_IDS.contains(idType);
    }

}
