from library.python import resource
from collections import namedtuple

import crypta.graph.soup.config.proto.source_type_pb2 as source_type
import crypta.graph.soup.config.proto.log_source_pb2 as log_source
import crypta.graph.soup.config.proto.ext_pb2 as ext
import crypta.lib.proto.identifiers.id_type_pb2 as id_type
import crypta.lib.proto.identifiers.ext_pb2 as ext_id

from crypta.graph.soup.config.proto.edge_type_pb2 import TEdgeProps as edge_props  # noqa
from crypta.graph.soup.config.proto.edge_type_pb2 import TEdges, TEdgeType, TEdgeTypeExt

import google.protobuf.text_format as tf

from google.protobuf.pyext._message import RepeatedScalarContainer


class ProtobufEnum(object):
    def __init__(self, enum, constructor, exts):
        self._values = dict()
        self._ignored = dict()
        self._by_type = dict()
        self._by_ext = dict()
        self._enum = enum

        for val_desc in enum.DESCRIPTOR.values:
            if val_desc.number <= 0:
                # we're not interested in dummy 'DEFAULT', 'UNKNOWN', ... etc. value
                # mark they with negative numbers
                self._ignored[val_desc.name] = val_desc.number
                continue

            opts = val_desc.GetOptions()
            val = constructor()
            val.Type = val_desc.number
            extension_name_value = None
            for ex in exts:
                extension_value = opts.Extensions[ex]

                if isinstance(extension_value, RepeatedScalarContainer):
                    getattr(val, ex.name).extend(extension_value)
                else:
                    setattr(val, ex.name, extension_value)

                if ex.name == "Name":
                    extension_name_value = extension_value

            self._values[val_desc.name] = val
            self._by_type[val_desc.number] = val
            if extension_name_value is not None:
                if extension_name_value in self._by_ext:
                    raise ValueError(
                        "Name extension should be unique. Duplicate '{name}'".format(name=extension_name_value)
                    )
                self._by_ext[extension_name_value] = val

    def values(self):
        return self._values.values()

    def by_type(self, typ):
        return self._by_type[typ]

    def by_name(self, name):
        return self._values[name]

    def by_ext(self, name):
        return self._by_ext[name]

    def __getattr__(self, attr):
        val = self._values.get(attr)
        if val:
            return val
        else:
            raise AttributeError('No %s named "%s"' % (self._enum.DESCRIPTOR.name, attr))


Edge = namedtuple("Edge", "Id1Type, Id2Type, SourceType, LogSource")


class Edges(object):
    def __init__(self, pb, id_types, source_types, log_types):
        self._values = dict()
        self._by_name = dict()

        edges = TEdges()
        tf.Merge(pb, edges)

        for rec in edges.Edges:
            et = rec.Type
            val = TEdgeTypeExt()
            val.Id1Type.CopyFrom(id_types.by_type(et.Id1Type))
            val.Id2Type.CopyFrom(id_types.by_type(et.Id2Type))
            val.SourceType.CopyFrom(source_types.by_type(et.SourceType))
            val.LogSource.CopyFrom(log_types.by_type(et.LogSource))
            val.Props.CopyFrom(rec.Props)
            val.Usage.CopyFrom(rec.Usage)

            self._values[self._mk_tuple(rec.Type)] = val
            self._by_name[(val.Id1Type.Name, val.Id2Type.Name, val.SourceType.Name, val.LogSource.Name)] = val

    def __iter__(self):
        for key, value in self._by_name.iteritems():
            yield Edge(*key), value

    def values(self):
        return self._values.values()

    def get_edge_type(self, id1, id2, st, ls):
        et = TEdgeType()
        et.Id1Type = id1.Type
        et.Id2Type = id2.Type
        et.SourceType = st.Type
        et.LogSource = ls.Type

        result = self._values.get(self._mk_tuple(et))
        if result is not None:
            return result
        else:
            et.Id1Type, et.Id2Type = et.Id2Type, et.Id1Type
            result = self._values.get(self._mk_tuple(et))
            if result is not None:
                return result

        raise ValueError("No such edge: (%s, %s, %s, %s)" % (id1.Name, id2.Name, st.Name, ls.Name))

    def get_edge_type_by_name(self, *args):
        return self._by_name[args]

    @classmethod
    def title(cls, *args):
        return "_".join(args)

    @classmethod
    def name(cls, et):
        return cls.title(*map(lambda item: item.Name, cls._mk_tuple(et)))

    @classmethod
    def tuplize(cls, et):
        """
            Protobuf objects aren't hashable, so this makes a hashable tuple for use in sets and dict keys
        """
        return tuple(map(lambda item: item.Type, cls._mk_tuple(et)))

    @classmethod
    def _mk_tuple(cls, typ):
        return Edge(typ.Id1Type, typ.Id2Type, typ.SourceType, typ.LogSource)


ID_TYPE = ProtobufEnum(id_type.EIdType, id_type.TIdType, [ext_id.Name, ext_id.RepObj])
SOURCE_TYPE = ProtobufEnum(source_type.ESourceType, source_type.TSourceType, [ext.Name, ext.Description])
LOG_SOURCE = ProtobufEnum(log_source.ELogSourceType, log_source.TLogSourceType, [ext.Name, ext.LogPath])

EDGE_TYPE = Edges(resource.find("/edge_types.pb.txt"), ID_TYPE, SOURCE_TYPE, LOG_SOURCE)
