package ru.yandex.crypta.graph2.model.soup.edge;

import java.util.Comparator;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Vec2;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph.soup.config.proto.ELogSourceType;
import ru.yandex.crypta.graph.soup.config.proto.ESourceType;
import ru.yandex.crypta.graph2.dao.yt.schema.extractor.CustomColumnType;
import ru.yandex.crypta.graph2.model.matching.proto.EdgeBetweenComponents;
import ru.yandex.crypta.graph2.model.soup.sources.ProtobufEnumLogSourceSerializer;
import ru.yandex.crypta.graph2.model.soup.sources.ProtobufEnumSourceTypeSerializer;
import ru.yandex.crypta.graph2.model.soup.vertex.ProtobufEnumIdTypeSerializer;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.inside.yt.kosher.impl.ytree.object.FieldsBindingStrategy;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeSerializerClass;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

@YTreeObject(bindingStrategy = FieldsBindingStrategy.ANNOTATED_ONLY)
public class Edge implements Comparable<Edge> {

    public static final ListF<String> EDGE_UNIQUE_KEY = EdgeType.EDGE_TYPE_UNIQUE_KEY.plus("id1", "id2");
    private static final Comparator<Edge> COMPARATOR = Comparator
            .comparing(Edge::calculateEdgeType)
            .thenComparing(Edge::getId1)
            .thenComparing(Edge::getId2);

    public static final Comparator<Edge> BY_WEIGHT_COMPARATOR = Comparator
            .<Edge>comparingDouble(e -> e.getSurvivalWeight().getOrElse(0.0))
            .thenComparingDouble(e -> e.getDatesWeight().getOrElse(0.0))
            .thenComparing(COMPARATOR);

    @YTreeField
    private final String id1;
    @YTreeField
    @YTreeSerializerClass(ProtobufEnumIdTypeSerializer.class)
    @CustomColumnType(ColumnValueType.STRING)
    private final EIdType id1Type;

    @YTreeField
    private final String id2;
    @YTreeField
    @YTreeSerializerClass(ProtobufEnumIdTypeSerializer.class)
    @CustomColumnType(ColumnValueType.STRING)
    private final EIdType id2Type;

    @YTreeField
    @YTreeSerializerClass(ProtobufEnumSourceTypeSerializer.class)
    @CustomColumnType(ColumnValueType.STRING)
    private final ESourceType sourceType;

    @YTreeField
    @YTreeSerializerClass(ProtobufEnumLogSourceSerializer.class)
    @CustomColumnType(ColumnValueType.STRING)
    private final ELogSourceType logSource;

    @YTreeField
    private ListF<String> dates;

    @YTreeField
    private Option<Double> datesWeight;
    @YTreeField
    private Option<Double> survivalWeight;

    @YTreeField
    private Option<Boolean> indevice;

    // pre-compute these to speed up
    private Vertex vertex1;
    private Vertex vertex2;
    private Vec2<Vertex> vertices;
    private EdgeType edgeType;

    public Edge(String id1, EIdType id1Type, String id2, EIdType id2Type,
                ESourceType sourceType, ELogSourceType logSource, ListF<String> dates, Option<Double> datesWeight,
                Option<Double> survivalWeight) {
        // Bender don't call this constructor
        // TODO: check if YTree calls it
        this.id1 = id1;
        this.id1Type = id1Type;
        this.id2 = id2;
        this.id2Type = id2Type;
        this.sourceType = sourceType;
        this.logSource = logSource;
        this.dates = dates;
        this.datesWeight = datesWeight;
        this.survivalWeight = survivalWeight;
        this.indevice = Option.empty();
    }

    public Edge(String id1, EIdType id1Type, String id2, EIdType id2Type,
                ESourceType sourceType, ELogSourceType logSource, ListF<String> dates) {
        this(id1, id1Type, id2, id2Type, sourceType, logSource, dates, Option.empty(), Option.empty());
    }

    public Edge(EdgeType edgeType, String id1, String id2) {
        this(edgeType, id1, id2, Cf.list());
    }

    public Edge(EdgeType edgeType, String id1, String id2, ListF<String> dates) {
        this(id1, edgeType.getId1Type(), id2, edgeType.getId2Type(),
                edgeType.getSourceType(), edgeType.getLogSource(), dates, Option.empty(), Option.empty());
    }

    public Edge(EdgeType edgeType, String id1, String id2, ListF<String> dates, Option<Double> weight1,
                Option<Double> weight2) {
        this(id1, edgeType.getId1Type(), id2, edgeType.getId2Type(),
                edgeType.getSourceType(), edgeType.getLogSource(), dates, weight1, weight2);
    }

    public Edge(EdgeBetweenComponents proto) {
        this(
                proto.getId1(),
                Soup.CONFIG.getIdType(proto.getId1Type()).getType(),
                proto.getId2(),
                Soup.CONFIG.getIdType(proto.getId2Type()).getType(),
                Soup.CONFIG.getSourceType(proto.getSourceType()).getType(),
                Soup.CONFIG.getLogSource(proto.getLogSource()).getType(),
                Cf.list(),
                Option.of(proto.getDatesWeight()),
                Option.of(proto.getSurvivalWeight())
        );
    }

    public Edge(ru.yandex.crypta.graph2.model.matching.proto.EdgeInComponent proto) {
        this(
                proto.getId1(),
                Soup.CONFIG.getIdType(proto.getId1Type()).getType(),
                proto.getId2(),
                Soup.CONFIG.getIdType(proto.getId2Type()).getType(),
                Soup.CONFIG.getSourceType(proto.getSourceType()).getType(),
                Soup.CONFIG.getLogSource(proto.getLogSource()).getType(),
                Cf.list(),
                Option.of(proto.getDatesWeight()),
                Option.of(proto.getSurvivalWeight())
        );
    }

    public Edge(ru.yandex.crypta.graph2.model.soup.proto.EnumEdge proto) {
        this(
                proto.getId1(),
                proto.getId1Type(),
                proto.getId2(),
                proto.getId2Type(),
                proto.getSourceType(),
                proto.getLogSource(),
                Cf.list(),
                Option.of(proto.getDatesWeight()),
                Option.of(proto.getSurvivalWeight())
        );
    }

    public Edge withDates(ListF<String> dates) {
        this.dates = dates;
        return this;
    }

    public EdgeType calculateEdgeType() {
        if (edgeType == null) {
            return edgeType = new EdgeType(id1Type, id2Type, sourceType, logSource);
        }
        return edgeType;
    }

    public boolean isLoop() {
        return id1.equals(id2) && id1Type.equals(id2Type);
    }

    public String getId1() {
        return id1;
    }

    public EIdType getId1Type() {
        return id1Type;
    }

    public String getId2() {
        return id2;
    }

    public EIdType getId2Type() {
        return id2Type;
    }

    public ESourceType getSourceType() {
        return sourceType;
    }

    public ELogSourceType getLogSource() {
        return logSource;
    }

    public ListF<String> getDates() {
        return dates;
    }

    public Option<Double> getDatesWeight() {
        return datesWeight;
    }

    public void setDatesWeight(double datesWeight) {
        this.datesWeight = Option.of(datesWeight);
    }

    public Option<Double> getSurvivalWeight() {
        return survivalWeight;
    }

    public void setSurvivalWeight(double survivalWeight) {
        this.survivalWeight = Option.of(survivalWeight);
    }

    public Option<Boolean> getIndevice() {
        return indevice;
    }

    public void setIndevice(Boolean indevice) {
        this.indevice = Option.of(indevice);
    }

    public void computeVertices() {
        vertex1 = new Vertex(id1, id1Type);
        vertex2 = new Vertex(id2, id2Type);
        vertices = new Vec2<>(vertex1, vertex2);
    }

    public Vec2<Vertex> getVertices() {
        if (vertices == null) {
            // TODO: when deserialized from YT, constructor is not called for some reason
            computeVertices();
        }
        return vertices;
    }

    public Vertex getVertex1() {
        if (vertices == null) {
            // TODO: when deserialized from YT, constructor is not called for some reason
            computeVertices();
        }
        return vertex1;
    }

    public Vertex getVertex2() {
        if (vertices == null) {
            // TODO: when deserialized from YT, constructor is not called for some reason
            computeVertices();
        }
        return vertex2;
    }

    public MultiEdgeKey getMultiEdgeKey() {
        return new MultiEdgeKey(id1, id1Type, id2, id2Type);
    }

    @Override
    public String toString() {
        return EdgeType.getTableName(id1Type, id2Type, sourceType, logSource)
                + "(" + id1 + " -> " + id2 + ")";
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Edge)) {
            return false;
        }

        Edge edge = (Edge) o;

        if (!id1.equals(edge.id1)) {
            return false;
        }
        if (id1Type != edge.id1Type) {
            return false;
        }
        if (!id2.equals(edge.id2)) {
            return false;
        }
        if (id2Type != edge.id2Type) {
            return false;
        }
        if (sourceType != edge.sourceType) {
            return false;
        }
        return logSource == edge.logSource;
    }

    @Override
    public int hashCode() {
        int result = id1.hashCode();
        result = 31 * result + id1Type.hashCode();
        result = 31 * result + id2.hashCode();
        result = 31 * result + id2Type.hashCode();
        result = 31 * result + sourceType.hashCode();
        result = 31 * result + logSource.hashCode();
        return result;
    }

    @Override
    public int compareTo(Edge other) {
        return COMPARATOR.compare(this, other);
    }
}
