package ru.yandex.crypta.graph2.model.soup.edge.weight.estimator;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeProps;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeRecord;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.model.soup.edge.EdgeProtoHelper;
import ru.yandex.crypta.graph2.model.soup.edge.EdgeTypeActivityStats;
import ru.yandex.crypta.graph2.model.soup.proto.Edge;
import ru.yandex.crypta.graph2.model.soup.proto.EdgeType;
import ru.yandex.crypta.graph2.model.soup.sources.EdgeTypeConfigProvider;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.lang.Check;

public class NormalizedDatesActivityEdgeWeightEstimator implements EdgeWeightEstimator {

    private static final Logger LOG = LoggerFactory.getLogger(NormalizedDatesActivityEdgeWeightEstimator.class);

    private EdgeTypeConfigProvider edgeTypeConfigProvider;

    private MapF<EdgeType, Double> meanDatesCountPerEdgeType = Cf.hashMap();
    private MapF<EdgeType, Double> meanEdgeLifetimePerEdgeType = Cf.hashMap();

    public NormalizedDatesActivityEdgeWeightEstimator(EdgeTypeConfigProvider edgeTypeConfigProvider,
                                                      CollectionF<EdgeTypeActivityStats> edgeTypesStats) {

        this.edgeTypeConfigProvider = edgeTypeConfigProvider;

        for (EdgeTypeActivityStats edgeTypeStats : edgeTypesStats) {
            EdgeType edgeType = EdgeProtoHelper.convertToProtoEdgeType(edgeTypeStats.getEdgeType());

            double meanDatesCount = getMeanOfHistogram(edgeTypeStats.getDatesCountHist());
            checkEmptyActivity(edgeType, meanDatesCount);
            meanDatesCountPerEdgeType.put(edgeType, meanDatesCount);
            meanDatesCountPerEdgeType.put(edgeType, meanDatesCount);

            double meanLifetime = getMeanOfHistogram(edgeTypeStats.getLifetimeDaysHist());
            checkEmptyActivity(edgeType, meanLifetime);
            meanEdgeLifetimePerEdgeType.put(edgeType, meanLifetime);
        }

    }

    public static NormalizedDatesActivityEdgeWeightEstimator buildFromYt(Dao dao, Option<YPath> table,
                                                                         EdgeTypeConfigProvider edgeTypeConfigProvider) {
        ListF<EdgeTypeActivityStats> stats;
        if (table.isPresent()) {
            YPath statsTable = table.get();
            LOG.info("Getting soup stats from {}", statsTable);
            stats = dao.ytTables()
                    .readYsonEntities(statsTable, EdgeTypeActivityStats.class);
        } else {
            LOG.info("Soup stats table is not found, using default");
            stats = Cf.list();
        }

        return new NormalizedDatesActivityEdgeWeightEstimator(edgeTypeConfigProvider, stats);

    }

    @Override
    public double getEdgeWeight(Edge edge) {

        EdgeType edgeType = EdgeProtoHelper.getEdgeType(edge);

        TEdgeRecord edgeTypeConfig = edgeTypeConfigProvider.getEdgeTypeConfig(edgeType);

        if (edgeTypeConfig.getProps().getEdgeStrength().equals(TEdgeProps.EEdgeStrength.PROBABILISTIC)) {
            return edge.getDatesWeight();
        }

        double baseWeight = edgeTypeConfigProvider.getBaseWeight(edgeType);

        if (edgeTypeConfigProvider.hasActivity(edgeType)) {
            List<String> dates = EdgeProtoHelper.getDates(edge);
            double edgeTypeActivity = dates.size() * baseWeight;

            double meanEdgeTypeActivity = getMeanEdgeTypeActivity(edgeType);
            checkEmptyActivity(edgeType, meanEdgeTypeActivity);

            return edgeTypeActivity / meanEdgeTypeActivity;
        } else {
            return baseWeight;
        }
    }

    private void checkEmptyActivity(EdgeType edgeType, double activity) {
        if (edgeTypeConfigProvider.hasActivity(edgeType)) {
            Check.isTrue(0.0 != activity && !Double.isInfinite(activity),
                    "Activity is empty. All empty activity edge types should be marked as hasActivity=false: ",
                    edgeType
            );
        }

    }

    public double getMeanEdgeTypeActivity(EdgeType edgeType) {
        return meanDatesCountPerEdgeType.getOrElse(edgeType, 1.0);
    }

    private double getMeanOfHistogram(MapF<Integer, Long> hist) {
        long totalEdges = hist.values().sum(Cf.Long);

        if (hist.isEmpty()) {
            return 0.0;
        }

        return hist.mapEntries(
                (datesCount, edgesCount) -> (double) edgesCount / (double) totalEdges * (double) datesCount
        ).sum(Cf.Double);
    }


    @Override
    public String toString() {
        return "NormalizedDatesActivityEdgeWeightEstimator{" +
                "edgeTypeConfigProvider=" + edgeTypeConfigProvider +
                ", meanDatesCountPerEdgeType=\n" + meanDatesCountPerEdgeType.entries().mkString("\n") +
                ", meanEdgeLifetimePerEdgeType=\n" + meanEdgeLifetimePerEdgeType.entries().mkString("\n") +
                '}';
    }

}
