package ru.yandex.crypta.graph2.model.soup.edge.weight.estimator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.model.soup.edge.EdgeProtoHelper;
import ru.yandex.crypta.graph2.model.soup.proto.Edge;
import ru.yandex.crypta.graph2.model.soup.proto.EdgeType;
import ru.yandex.crypta.graph2.model.soup.sources.EdgeTypeConfigProvider;
import ru.yandex.inside.yt.kosher.cypress.YPath;


public class SurvivalEdgeWeightEstimator implements EdgeWeightEstimator {

    private static final Logger LOG = LoggerFactory.getLogger(SurvivalEdgeWeightEstimator.class);

    private final SurvivalEdgeModel survivalEdgeModel;
    private final EdgeTypeConfigProvider edgeTypeConfigProvider;


    public SurvivalEdgeWeightEstimator(SurvivalEdgeModel survivalEdgeModel, EdgeTypeConfigProvider edgeTypeConfigProvider) {
        this.survivalEdgeModel = survivalEdgeModel;
        this.edgeTypeConfigProvider = edgeTypeConfigProvider;
    }

    public static SurvivalEdgeWeightEstimator buildFromYt(Dao dao, Option<YPath> table,
                                                          EdgeTypeConfigProvider edgeTypeConfigProvider) {
        SurvivalEdgeModel survivalEdgeModel;
        if (table.isPresent()) {
            LOG.info("Reading survival model from {}", table.get());
            survivalEdgeModel = new SurvivalEdgeModel(dao, table.get());
        } else {
            LOG.info("Survival model table is not found, using default");
            survivalEdgeModel = new SurvivalEdgeModel();
        }

        return new SurvivalEdgeWeightEstimator(survivalEdgeModel, edgeTypeConfigProvider);
    }

    @Override
    public double getEdgeWeight(Edge edge) {
        EdgeType edgeType = EdgeProtoHelper.getEdgeType(edge);
        if (edgeTypeConfigProvider.isStrong(edgeType)) {
            return 1.;
        }
        if (edge.getSurvivalWeight() > Double.MIN_VALUE && edge.getSurvivalWeight() <= 1) {
            return edge.getSurvivalWeight();
        }
        double edgeWeight = survivalEdgeModel.getEdgeWeight(edge);
        return Double.isNaN(edgeWeight) ? 0.0 : edgeWeight;
    }
}
