package ru.yandex.crypta.graph2.model.soup.props;

import java.util.Optional;
import java.util.function.Consumer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph2.dao.yt.bendable.YsonMultiEntitySupport;
import ru.yandex.crypta.graph2.model.id.proto.IdInfo;
import ru.yandex.crypta.graph2.model.soup.props.info.ExactSocdem;
import ru.yandex.crypta.graph2.model.soup.props.info.ProbSocdem;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

public class VertexPropertiesCollector extends YsonMultiEntitySupport implements Consumer<YTreeMapNode> {

    private MapF<Vertex, YTreeMapNode> rawRecs = Cf.hashMap();
    private MapF<Vertex, Yandexuid> yuids = Cf.hashMap();
    private MapF<Vertex, DeviceId> deviceIds = Cf.hashMap();
    private MapF<Vertex, Uuid> uuids = Cf.hashMap();
    private MapF<Vertex, ListF<ExactSocdem>> exactSocdems = Cf.hashMap();
    private MapF<Vertex, ProbSocdem> probSocdems = Cf.hashMap();
    private MapF<Vertex, CommonShared> identifiers = Cf.hashMap();
    private MapF<Vertex, Outlier> outliers = Cf.hashMap();

    public VertexPropertiesCollector() {
    }

    public VertexPropertiesCollector(MapF<Vertex, Yandexuid> yuids,
                                     MapF<Vertex, DeviceId> deviceIds,
                                     MapF<Vertex, Uuid> uuids,
                                     MapF<Vertex, ListF<ExactSocdem>> exactSocdems,
                                     MapF<Vertex, ProbSocdem> probSocdems) {
        this.yuids = yuids;
        this.deviceIds = deviceIds;
        this.uuids = uuids;
        this.exactSocdems = exactSocdems;
        this.probSocdems = probSocdems;
    }

    private Option<Integer> getMainRegion(IdInfo proto) {
        int region = (int)proto.getMainRegion();
        if (region == 0) {
            return Option.empty();
        } else {
            return Option.of(region);
        }
    }

    static private Option<Integer> getYearOfBirth(int year) {
        if (year == 0) {
            return Option.empty();
        } else {
            return Option.of(year);
        }
    }

    static private Option<Integer> getYearOfBirth(IdInfo proto) {
        return getYearOfBirth((int)proto.getYearOfBirth());
    }

    static private Option<String> getSocdemSource(String socdemSource) {
        if (socdemSource.isEmpty()) {
            return Option.empty();
        } else {
            return Option.of(socdemSource);
        }
    }

    static private Option<String> getSocdemSource(IdInfo proto) {
        return getSocdemSource(proto.getSocdemSource());
    }

    public static ExactSocdem getExactSocdem(String genderString, int year,  String socdemSource) {
        ExactSocdem.Gender gender = ExactSocdem.Gender.COMPATIBILITY_MAP.getOrElse(genderString, null);
        Option<ExactSocdem.Gender> optionGender = (gender == null) ? Option.empty() : Option.of(gender);
        return new ExactSocdem(
                getYearOfBirth(year),
                optionGender,
                getSocdemSource(socdemSource)
        );
    }

    private static ExactSocdem getExactSocdem(IdInfo proto) {
        ExactSocdem.Gender gender = ExactSocdem.Gender.COMPATIBILITY_MAP.getOrElse(proto.getGender(), null);
        Option<ExactSocdem.Gender> optionGender = (gender == null) ? Option.empty() : Option.of(gender);
        return new ExactSocdem(
                getYearOfBirth(proto),
                optionGender,
                getSocdemSource(proto)
        );
    }

    private static ProbSocdem getProbSocdem(IdInfo proto) {
        return new ProbSocdem(Cf.map(
                ExactSocdem.Gender.MALE.value(), proto.getProbMale(),
                ExactSocdem.Gender.FEMALE.value(), proto.getProbFemale()
        ), Cf.map());
    }

    public void accept(IdInfo proto) {
        String source = proto.getSource();
        String cryptaId = proto.getCryptaId();
        Vertex vertex = new Vertex(
                proto.getId(),
                proto.getIdType()
        );

        EIdType idType = vertex.getIdType();

        if (idType == EIdType.YANDEXUID && source.equals(Yandexuid.YUID_WITH_ALL_SOURCE)) {
            Yandexuid yandexuid = new Yandexuid(
                    vertex,
                    cryptaId,
                    proto.getUaProfile(),
                    getMainRegion(proto),
                    source,
                    proto.getIsActive());
            yuids.put(vertex, yandexuid);
            rawRecs.put(vertex, serialize(yandexuid));

        } else if (Soup.CONFIG.isDeviceIdMainId(idType) && source.equals(DeviceId.APP_METRICA_SOURCE)) {
            DeviceId deviceId = new DeviceId(
                    vertex,
                    cryptaId,
                    proto.getUaProfile(),
                    proto.getDeviceType(),
                    proto.getOs(),
                    getMainRegion(proto),
                    source,
                    proto.getIsActive()
            );
            deviceIds.put(vertex, deviceId);
            rawRecs.put(vertex, serialize(deviceId));

        } else if (idType == EIdType.UUID && source.equals(DeviceId.APP_METRICA_SOURCE)) {
            Uuid uuid =  new Uuid(
                    vertex,
                    cryptaId,
                    proto.getAppId(),
                    source,
                    proto.getIsActive()
            );
            uuids.put(vertex, uuid);
            rawRecs.put(vertex, serialize(uuid));
        } else if (source.equals(CommonShared.COMMON_SHARED_SOURCE)) {
            CommonShared identifier = new CommonShared(
                    vertex,
                    cryptaId,
                    source,
                    Cf.list()
            );
            identifiers.put(vertex, identifier);
            rawRecs.put(vertex, serialize(identifier));
        } else if (source.equals(Outlier.OUTLIER_SOURCE)) {
            Outlier identifier = new Outlier(
                    vertex,
                    cryptaId,
                    source,
                    proto.getId(),
                    proto.getIdType(),
                    proto.getAnomalyScore()
            );
            outliers.put(vertex, identifier);
            rawRecs.put(vertex, serialize(identifier));
        }

        if (VertexExactSocdem.EXACT_SOCDEM_SOURCE.equals(source)) {
            ExactSocdem exactSocdem = getExactSocdem(proto);
            VertexExactSocdem vSocdem = new VertexExactSocdem(
                    vertex,
                    cryptaId,
                    exactSocdem,
                    source,
                    exactSocdem.getSocdemSource()
            );
            exactSocdems.putIfAbsent(vertex, Cf.arrayList());
            exactSocdems.getTs(vertex).add(vSocdem.getSocdem());
            rawRecs.put(vertex, serialize(vSocdem));
        }
        if (VertexSocdemProbSegments.SOCDEM_SOURCE.equals(source)) {
            ProbSocdem probSocdem = getProbSocdem(proto);
            probSocdems.put(vertex, probSocdem);
        }
    }

    @Override
    public void accept(YTreeMapNode node) {
        Optional<String> sourceO = node.getStringO(VertexProperties.SOURCE_COLUMN);
        if (!sourceO.isPresent()) {
            return;
        }
        String source = sourceO.get();

        Vertex vertex = parse(node, Vertex.class);
        resetTableIndex(node);
        rawRecs.put(vertex, node);

        EIdType idType = vertex.getIdType();

        if (idType == EIdType.YANDEXUID && source.equals(Yandexuid.YUID_WITH_ALL_SOURCE)) {
            Yandexuid yandexuid = parse(node, Yandexuid.class);
            yuids.put(vertex, yandexuid);

        } else if (Soup.CONFIG.isDeviceIdMainId(idType) && source.equals(DeviceId.APP_METRICA_SOURCE)) {
            DeviceId deviceId = parse(node, DeviceId.class);
            deviceIds.put(vertex, deviceId);

        } else if (idType == EIdType.UUID && source.equals(DeviceId.APP_METRICA_SOURCE)) {
            Uuid uuid = parse(node, Uuid.class);
            uuids.put(vertex, uuid);
        } else if (source.equals(CommonShared.COMMON_SHARED_SOURCE)) {
            CommonShared identifier = parse(node, CommonShared.class);
            identifiers.put(vertex, identifier);
        } else if (source.equals(Outlier.OUTLIER_SOURCE)) {
            Outlier identifier = parse(node, Outlier.class);
            outliers.put(vertex, identifier);
        }

        if (VertexExactSocdem.EXACT_SOCDEM_SOURCE.equals(source)) {
            VertexExactSocdem vSocdem = parse(node, VertexExactSocdem.class);
            exactSocdems.putIfAbsent(vertex, Cf.arrayList());
            exactSocdems.getTs(vertex).add(vSocdem.getSocdem());
        }

    }

    public MapF<Vertex, YTreeMapNode> getRawRecs() {
        return rawRecs;
    }

    public MapF<Vertex, Yandexuid> getYuids() {
        return yuids;
    }

    public MapF<Vertex, DeviceId> getDeviceIds() {
        return deviceIds;
    }

    public MapF<Vertex, Uuid> getUuids() {
        return uuids;
    }

    public MapF<Vertex, ListF<ExactSocdem>> getExactSocdems() {
        return exactSocdems;
    }

    public MapF<Vertex, ProbSocdem> getProbSocdems() {
        return probSocdems;
    }

    public MapF<Vertex, CommonShared> getSharedIds() {
        return identifiers;
    }

    public void addYandexuid(Yandexuid yuid) {
        this.yuids.put(yuid.getVertex(), yuid);
    }

    public void addDeviceId(DeviceId deviceId) {
        this.deviceIds.put(deviceId.getVertex(), deviceId);
    }

    public boolean isActive(Vertex vertex) {

        Option<? extends HasActivity> id;
        if (vertex.getIdType() == EIdType.YANDEXUID) {
            id = getYuids().getO(vertex);
        } else if (vertex.getIdType() == EIdType.UUID) {
            id = getUuids().getO(vertex);
        } else if (Soup.CONFIG.isDeviceIdMainId(vertex.getIdType())) {
            id = getDeviceIds().getO(vertex);
        } else {
            throw new IllegalStateException("Only vertices of HasActivity types are supported: " + vertex.getIdType());
        }

        return id.isPresent() && id.get().isActive();
    }

    public MapF<Vertex, Outlier> getOutliers() {
        return outliers;
    }

    public void setOutliers(MapF<Vertex, Outlier> outliers) {
        this.outliers = outliers;
    }
}
