package ru.yandex.crypta.graph2.model.soup.vertex;

import java.util.Comparator;
import java.util.Objects;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph2.dao.yt.schema.extractor.CustomColumnType;
import ru.yandex.crypta.graph2.model.matching.proto.VertexInComponent;
import ru.yandex.crypta.graph2.model.soup.proto.EnumVertex;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.TIdType;
import ru.yandex.inside.yt.kosher.impl.operations.utils.YtSerializable;
import ru.yandex.inside.yt.kosher.impl.ytree.object.FieldsBindingStrategy;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeSerializerClass;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

@YTreeObject(bindingStrategy = FieldsBindingStrategy.ANNOTATED_ONLY)
public class Vertex implements Comparable<Vertex>, YtSerializable {

    public static final String ID_COLUMN = "id";
    public static final String ID_TYPE_COLUMN = "id_type";
    public static final ListF<String> VERTEX_REDUCE_KEY = Cf.list(ID_COLUMN, ID_TYPE_COLUMN);
    public static final ListF<String> ID_TYPE_FIRST_KEY = Cf.list(ID_TYPE_COLUMN, ID_COLUMN);

    public static final Comparator<Vertex> COMPARATOR = Comparator
            .comparing(Vertex::getId)
            .thenComparing(v -> Soup.CONFIG.name(v.getIdType()));


    @YTreeField(key = ID_COLUMN)
    private String id;
    @YTreeField(key = ID_TYPE_COLUMN)
    @YTreeSerializerClass(ProtobufEnumIdTypeSerializer.class)
    @CustomColumnType(ColumnValueType.STRING)
    private EIdType idType;

    public Vertex(String id, EIdType idType) {
        this.id = id;
        this.idType = idType;
    }

    public Vertex(String id, String idType) {
        this.id = id;

        TIdType tIdType = Soup.CONFIG.getIdType(idType);
        Objects.requireNonNull(tIdType, () -> "Unknown EIdType: " + idType);
        this.idType = tIdType.getType();
    }

    public Vertex(EnumVertex vertexKey) {
        this(vertexKey.getId(), vertexKey.getIdType());
    }

    public Vertex(VertexInComponent vertexKey) {
        this(vertexKey.getId(), vertexKey.getIdType());
    }

    public String getId() {
        return id;
    }

    public EIdType getIdType() {
        return idType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Vertex vertex)) {
            return false;
        }

        if (!id.equals(vertex.id)) {
            return false;
        }
        return idType == vertex.idType;
    }

    @Override
    public int hashCode() {
        int result = id.hashCode();
        result = 31 * result + idType.hashCode();
        return result;
    }

    public String repr() {
        return id + "(" + idType + ")";
    }

    @Override
    public String toString() {
        return id + "(" + idType + ")";
    }

    @Override
    public int compareTo(Vertex o) {
        return COMPARATOR.compare(this, o);
    }


}
