package ru.yandex.crypta.graph2.soup;

import java.time.LocalDate;

import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yql.Yql;
import ru.yandex.crypta.graph2.dao.yql.YqlConfig;
import ru.yandex.crypta.graph2.dao.yt.YtConfig;
import ru.yandex.crypta.graph2.dao.yt.ops.YtOpsParams;
import ru.yandex.crypta.graph2.soup.config.ConfigParser;
import ru.yandex.crypta.graph2.soup.config.PrepareSoupConfig;
import ru.yandex.crypta.graph2.soup.config.SoupAndStorageProcessingParams;
import ru.yandex.crypta.graph2.soup.workflow.PrepareSoupWorkflow;
import ru.yandex.crypta.graph2.soup.workflow.PrepareSoupWorkflow.ProcessingDates;
import ru.yandex.crypta.graph2.utils.NativeLibHelper;
import ru.yandex.crypta.graph2.utils.YamlConfig;
import ru.yandex.crypta.graph2.workflow.EmptyInput;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class PrepareSoupMain {

    private static final Logger LOG = LoggerFactory.getLogger(PrepareSoupMain.class);

    public static void main(String[] args) {

        OptionParser parser = new OptionParser();
        OptionSpec<String> configArg = parser
                .accepts("config")
                .withRequiredArg()
                .defaultsTo("config.yaml");

        parser.allowsUnrecognizedOptions();
        OptionSet options = parser.parse(args);

        NativeLibHelper.setLocalJavaLibraryPath();

        Yt yt = YtConfig.getYt(true);
        Yql yql = YqlConfig.getYql();
        YtOpsParams ytOpsParams = new YtOpsParams();

        Dao dao = new Dao(yt, yql, ytOpsParams);

        LOG.info("Set yt, yql env.");

        String configFile = options.valueOf(configArg);

        PrepareSoupConfig config = YamlConfig.readConfigFromFile(configFile, PrepareSoupConfig.class);
        LOG.info("Config is received from file {}", configFile);
        LOG.info("PrepareSoupConfig:\n{}", config);

        SoupAndStorageProcessingParams params = ConfigParser.parse(config);
        LOG.info("SoupAndStorageProcessingParams:\n{}", params);

        ProcessingDates processingDates = getSoupGenerateDates(dao, config);
        PrepareSoupWorkflow prepareSoupWorkflow = new PrepareSoupWorkflow(dao, params, processingDates);

        LOG.info("Start running");
        prepareSoupWorkflow.run(EmptyInput.INSTANCE);
    }

    private static ProcessingDates getSoupGenerateDates(Dao dao, PrepareSoupConfig config) {
        LocalDate startDate = LocalDate.now();

        LocalDate soupGenerateDate = dao.ytCypress().getGenerationDate(
                YPath.simple(config.getSoupDir())
        ).getOrElse(startDate);

        LocalDate idsStorageGenerateDate = dao.ytCypress().getGenerationDate(
                YPath.simple(config.getIdsStorageDir())
        ).getOrElse(startDate);

        LOG.info("soupGenerateDate: " + soupGenerateDate);
        LOG.info("idsStorageGenerateDate: " + idsStorageGenerateDate);
        return new ProcessingDates(soupGenerateDate, idsStorageGenerateDate);
    }
}
