package ru.yandex.crypta.graph2.soup.workflow;

import java.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.soup.config.SoupAndStorageProcessingParams;
import ru.yandex.crypta.graph2.soup.workflow.PrepareVerticesPropertiesTask.IdsStorageDirs;
import ru.yandex.crypta.graph2.workflow.EmptyInput;
import ru.yandex.crypta.graph2.workflow.Task;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class PrepareSoupWorkflow extends Task<EmptyInput, ListF<YPath>, SoupAndStorageProcessingParams> {

    private final PrepareSoupTask prepareSoupTask;
    private final PrepareVerticesPropertiesTask prepareVerticesPropertiesTask;
    private final CalculateSoupStatsTask calculateSoupStatsTask;
    private final DumpSoupConfigTask dumpSoupConfigTask;
    private final ProcessingDates processingDates;

    public PrepareSoupWorkflow(Dao dao, SoupAndStorageProcessingParams params, ProcessingDates processingDates) {
        super(dao, params.getOutputDir(), params);

        this.processingDates = processingDates;

        this.prepareSoupTask = new PrepareSoupTask(
                dao, workdir,
                params,
                processingDates
        );
        this.prepareVerticesPropertiesTask = new PrepareVerticesPropertiesTask(
                dao, workdir,
                params
        );
        this.calculateSoupStatsTask = new CalculateSoupStatsTask(
                dao, workdir.child("stats"),
                params
        );
        this.dumpSoupConfigTask = new DumpSoupConfigTask(
                dao, workdir,
                params
        );
    }

    @Override
    protected void runImpl(EmptyInput emptyInput) {

        dumpSoupConfigTask.run(EmptyInput.INSTANCE);

        prepareSoupTask.run(EmptyInput.INSTANCE);

        prepareVerticesPropertiesTask.run(new IdsStorageDirs(
                params.getIdsStorageDir().get(),
                params.getSocdemStorageDir().get(),
                params.getProbSocdemTable().get(),
                params.getSharedStorageDir().get(),
                params.getOutliersStorageDir().get()
        ));

        calculateSoupStatsTask.run(prepareSoupTask.getOutput().first());


        LocalDate endOfProcessingDate = LocalDate.now();

        getOutput().forEach(t -> {
            processingDates.setDates(dao, t, endOfProcessingDate);
        });
    }

    @Override
    public ListF<YPath> getOutput() {
        return Cf.list(
                prepareVerticesPropertiesTask.getOutput(),
                calculateSoupStatsTask.getOutput()
        ).plus(prepareSoupTask.getOutput());
    }

    @Override
    public String getDescription() {
        return "Prepares many soup and ids storage tables and calculate soup stats";
    }

    public static class ProcessingDates {
        final LocalDate idsStorageGenerateDate;
        final LocalDate soupGenerateDate;

        public ProcessingDates(LocalDate idsStorageGenerateDate, LocalDate soupGenerateDate) {
            this.idsStorageGenerateDate = idsStorageGenerateDate;
            this.soupGenerateDate = soupGenerateDate;
        }

        public void setDates(Dao dao, Option<GUID> txId, YPath path) {
            LocalDate endOfProcessingDate = LocalDate.now();
            dao.ytCypress().setRunDate(txId, path, endOfProcessingDate);
            dao.ytCypress().setGenerationDate(txId, path, soupGenerateDate);
        }

        public void setDates(Dao dao, YPath path, LocalDate endOfProcessingDate) {
            dao.ytCypress().setRunDate(path, endOfProcessingDate);
            dao.ytCypress().setGenerationDate(path, soupGenerateDate);
        }
    }
}
