package ru.yandex.crypta.graph2.soup.workflow;

import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.model.soup.props.VertexPropertiesSchemaHelper;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.graph2.soup.config.SoupAndStorageProcessingParams;
import ru.yandex.crypta.graph2.soup.workflow.PrepareVerticesPropertiesTask.IdsStorageDirs;
import ru.yandex.crypta.graph2.soup.workflow.ops.ImportIdStorageVerticesProperties;
import ru.yandex.crypta.graph2.soup.workflow.ops.ImportProbSocdemStorageVerticesProperties;
import ru.yandex.crypta.graph2.soup.workflow.ops.ImportSharedStorageVerticesProperties;
import ru.yandex.crypta.graph2.soup.workflow.ops.ImportSocdemStorageVerticesProperties;
import ru.yandex.crypta.graph2.workflow.Task;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.TIdType;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.yt.ytclient.tables.TableSchema;

public class PrepareVerticesPropertiesTask extends Task<IdsStorageDirs, YPath, SoupAndStorageProcessingParams> {

    private static final Logger LOG = LoggerFactory.getLogger(PrepareVerticesPropertiesTask.class);

    private static final Set<String> ALL_POSSIBLE_TYPES = Soup.CONFIG
            .getIdTypes()
            .getAll()
            .stream()
            .map(TIdType::getName)
            .collect(Collectors.toSet());

    private YPath verticesPropertiesMerged;

    public PrepareVerticesPropertiesTask(Dao dao, YPath workdir, SoupAndStorageProcessingParams params) {
        super(dao, workdir, params);

        this.verticesPropertiesMerged = workdir.child("vertices_properties");
    }


    private ListF<YPath> getAllSecondLevelTables(YPath basePath, Set<String> filterIdTypes) {
        if (!dao.yt().cypress().exists(basePath)) {
            return Cf.list();
        }

        return dao.ytCypress().ls(basePath)
                .filter(
                        dir -> filterIdTypes.contains(dir.name())
                )
                .flatMap(
                        firstLevelDir -> dao.ytCypress().ls(firstLevelDir)
                );
    }

    private Option<YPath> importSharedStorage(YPath sharedStorageDir) {
        ListF<YPath> sharedVps = Cf.list(
                sharedStorageDir.child("common_shared")
        );
        LOG.info("Found vertices properties in shared storage: {}", sharedVps);

        if (!sharedVps.isEmpty()) {
            TableSchema schema = VertexPropertiesSchemaHelper.getUnionSharedPropertiesSchema();
            YPath sharedPropertiesTable = dao.ytCypress()
                    .createTableWithSchema(workdir.child("shared_properties"), schema);
            LOG.info("Preparing shared storage properties...");
            var op = dao.ytOps().mapOperation(
                    sharedVps,
                    Cf.list(sharedPropertiesTable),
                    new ImportSharedStorageVerticesProperties()
            );

            op.getMapperSpecBuilder().setMemoryLimit(DataSize.fromGigaBytes(1));
            op.runSync();

            return Option.of(sharedPropertiesTable);
        } else {
            return Option.empty();
        }
    }

    private Option<YPath> importSocdemStorage(YPath socdemStorageDir) {
        ListF<YPath> socdemVps = getAllSecondLevelTables(socdemStorageDir, ALL_POSSIBLE_TYPES);
        LOG.info("Found vertices properties in socdem storage: {}", socdemVps);

        if (!socdemVps.isEmpty()) {
            TableSchema schema = VertexPropertiesSchemaHelper.getUnionSocdemPropertiesSchema();
            YPath socdemPropertiesTable = dao.ytCypress()
                    .createTableWithSchema(workdir.child("socdem_properties"), schema);
            LOG.info("Preparing socdem storage properties...");
            dao.ytOps().mapSync(
                    socdemVps,
                    Cf.list(socdemPropertiesTable),
                    new ImportSocdemStorageVerticesProperties()
            );

            return Option.of(socdemPropertiesTable);
        } else {
            return Option.empty();
        }
    }

    private Option<YPath> importProbSocdemStorage(YPath probSocdemTable) {
        if (probSocdemTable == null || !dao.yt().cypress().exists(probSocdemTable)) {
            return Option.empty();
        }
        TableSchema schema = VertexPropertiesSchemaHelper.getUnionSocdemPropertiesSchema();
        YPath probSocdemPropertiesTable = dao.ytCypress()
                .createTableWithSchema(workdir.child("prob_socdem_properties"), schema);
        LOG.info("Preparing prob socdem");
        dao.ytOps().mapSync(
                Cf.list(probSocdemTable),
                Cf.list(probSocdemPropertiesTable),
                new ImportProbSocdemStorageVerticesProperties()
        );

        return Option.of(probSocdemPropertiesTable);
    }

    private Option<YPath> importIdsStorage(YPath idsStorageDir) {
        ListF<YPath> idStorageVps = Cf.list(
                idsStorageDir
                        .child(Soup.CONFIG.name(EIdType.YANDEXUID))
                        .child("yuid_with_all_info"),
                idsStorageDir
                        .child(Soup.CONFIG.name(EIdType.OLD_DEVICE_ID))
                        .child("app_metrica_month"),
                idsStorageDir
                        .child(Soup.CONFIG.name(EIdType.UUID))
                        .child("app_metrica_month")
        );

        LOG.info("Found vertices properties in ids storage: {}", idStorageVps);

        if (!idStorageVps.isEmpty()) {
            TableSchema schema = VertexPropertiesSchemaHelper.getUnionIdsPropertiesSchema();
            YPath idsStoragePropertiesTable = dao.ytCypress()
                    .createTableWithSchema(workdir.child("ids_storage_properties"), schema);
            LOG.info("Preparing ids storage properties...");

            var op = dao.ytOps().mapOperation(
                    idStorageVps,
                    Cf.list(idsStoragePropertiesTable),
                    new ImportIdStorageVerticesProperties()
            );

            op.getMapperSpecBuilder().setMemoryLimit(DataSize.fromGigaBytes(1));
            op.runSync();

            return Option.of(idsStoragePropertiesTable);
        } else {
            return Option.empty();
        }

    }

    @Override
    public void runImpl(IdsStorageDirs idsStorageDirs) {
        dao.ytCypress().ensureDir(workdir);

        // TODO: single table processing
        if (params.getIdsStorageTable().isPresent()) {
            YPath verticesPropertiesTable = params.getIdsStorageTable().get();
            dao.ytOps().sortSync(verticesPropertiesTable, Vertex.VERTEX_REDUCE_KEY);
        } else {

            Option<YPath> idsStorage = importIdsStorage(idsStorageDirs.idsStorageDir);
            Option<YPath> socdemStorage = importSocdemStorage(idsStorageDirs.socdemStorageDir);
            Option<YPath> probSocdemStorage = importProbSocdemStorage(idsStorageDirs.probSocdemTable);
            Option<YPath> sharedStorage = importSharedStorage(idsStorageDirs.sharedStorageDir);
            Option<YPath> outliersStorage = Option.of(idsStorageDirs.outliersStorageDir);

            ListF<YPath> allVPTables = idsStorage.plus(socdemStorage)
                    .plus(sharedStorage).plus(outliersStorage).plus(probSocdemStorage);

            dao.ytTr().withTransactionId(trId -> {

                dao.ytCypress().createTableWithSchema(
                        trId,
                        verticesPropertiesMerged,
                        VertexPropertiesSchemaHelper
                                .getUnionVertexPropertiesSchema()
                                .toBuilder()
                                .sortBy(Vertex.VERTEX_REDUCE_KEY)
                                .build()
                );

                if (allVPTables.isEmpty()) {
                    LOG.warn("Not found any vertices properties");
                } else {
                    LOG.info("Preparing vertices properties...");
                    dao.ytOps().sortOperation(allVPTables, verticesPropertiesMerged, Vertex.VERTEX_REDUCE_KEY)
                            .withSchemaInferenceMode("from_output")
                            .runSync(trId);

                    LOG.info("Preparing vertices properties... DONE");
                }
            });
        }
    }

    @Override
    public YPath getOutput() {
        if (params.getSoupTable().isPresent()) {
            return params.getIdsStorageTable().get();
        } else {
            return verticesPropertiesMerged;
        }
    }

    @Override
    public String getDescription() {
        return "Prepares all vertices properties from ids storage";
    }

    public static class IdsStorageDirs {
        YPath idsStorageDir;
        YPath socdemStorageDir;
        YPath probSocdemTable;
        YPath sharedStorageDir;
        YPath outliersStorageDir;

        public IdsStorageDirs(YPath idsStorageDir, YPath socdemStorageDir, YPath probSocdemTable, YPath sharedStorageDir,
                              YPath outliersStorageDir) {
            this.idsStorageDir = idsStorageDir;
            this.socdemStorageDir = socdemStorageDir;
            this.probSocdemTable = probSocdemTable;
            this.sharedStorageDir = sharedStorageDir;
            this.outliersStorageDir = outliersStorageDir;
        }
    }
}
