package ru.yandex.crypta.graph2.soup.workflow.ops;

import java.time.LocalDate;
import java.util.List;

import ru.yandex.crypta.graph2.model.soup.edge.EdgeProtoHelper;
import ru.yandex.crypta.graph2.model.soup.edge.weight.estimator.EdgeWeightEstimator;
import ru.yandex.crypta.graph2.model.soup.proto.Edge;
import ru.yandex.crypta.graph2.model.soup.proto.EdgeType;
import ru.yandex.crypta.graph2.model.soup.sources.EdgeTypeConfigProvider;
import ru.yandex.inside.yt.kosher.impl.operations.utils.YtSerializable;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.operations.map.Mapper;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

public class PrepareSoupEdges implements Mapper<Edge, Edge>, YtSerializable {

    private final EdgeTypeConfigProvider edgeTypeConfigProvider;
    private final EdgeWeightEstimator datesEdgeWeightEstimator;
    private final EdgeWeightEstimator survivalEdgeWeightEstimator;
    private final LocalDate soupPrepateDate;
    private final int soupTablesCount;

    private static final int PREPARE_SOUP_OUPUT_TABLES_COUNT = 2;

    private static final int TO_SOUP = 0;
    private static final int TO_DECAYED = 1;

    public PrepareSoupEdges(EdgeTypeConfigProvider edgeTypeConfigProvider,
                            EdgeWeightEstimator datesEdgeWeightEstimator,
                            EdgeWeightEstimator survivalEdgeWeightEstimator,
                            LocalDate soupPrepateDate,
                            int soupTablesCount) {
        this.edgeTypeConfigProvider = edgeTypeConfigProvider;
        this.datesEdgeWeightEstimator = datesEdgeWeightEstimator;
        this.survivalEdgeWeightEstimator = survivalEdgeWeightEstimator;
        this.soupPrepateDate = soupPrepateDate;
        this.soupTablesCount = soupTablesCount;
    }

    private boolean IsEdgeDecayed(Edge edge, EdgeType edgeType) {
        var matchingDaysToLive = edgeTypeConfigProvider.getMatchingDaysToLive(edgeType);
        if (matchingDaysToLive == 0) {
            return false;
        }

        List<String> dates = EdgeProtoHelper.getDates(edge);
        if (dates.isEmpty()) { // XXX what should we do with decaying edges with no days of activity
            return true;
        }

        var lastActiveDate = LocalDate.parse(dates.get(dates.size() - 1));
        if (lastActiveDate.plusDays(matchingDaysToLive).isBefore(soupPrepateDate)) {
            return true;
        }

        return false;
    }

    @Override
    public void map(Edge edge, Yield<Edge> yield, Statistics statistics) {
        Edge.Builder weightedEdgeBuilder = edge.toBuilder();

        EdgeType edgeType = EdgeProtoHelper.getEdgeType(edge);
        if (edgeTypeConfigProvider.isStrong(edgeType)) {
            weightedEdgeBuilder
                    .setDatesWeight(1.0)
                    .setSurvivalWeight(1.0);
        } else {
            weightedEdgeBuilder
                    .setDatesWeight(datesEdgeWeightEstimator.getEdgeWeight(edge))
                    .setSurvivalWeight(survivalEdgeWeightEstimator.getEdgeWeight(edge));
        }

        var tableIndex = IsEdgeDecayed(edge, edgeType) ? TO_DECAYED : TO_SOUP;
        yield.yield(tableIndex, weightedEdgeBuilder.build());
    }

    @Override
    public YTableEntryType<Edge> inputType() {
        return YTableEntryTypes.nativeProto(Edge.newBuilder(), soupTablesCount);
    }

    @Override
    public YTableEntryType<Edge> outputType() {
        return YTableEntryTypes.nativeProto(Edge.newBuilder(), PREPARE_SOUP_OUPUT_TABLES_COUNT);
    }
}
