package ru.yandex.crypta.graph2.soup.dynamic.workflow;

import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.ops.Await;
import ru.yandex.crypta.graph2.dao.yt.ops.MapOperation;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.soup.dynamic.config.SoupDynamicProcessingParams;
import ru.yandex.crypta.graph2.soup.dynamic.workflow.ops.PrepareSoupEdgesBackwards;
import ru.yandex.crypta.graph2.workflow.EmptyInput;
import ru.yandex.crypta.graph2.workflow.Task;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

public class PrepareSoupEdgesBackwardsTask extends Task<EmptyInput, YPath, SoupDynamicProcessingParams> {

    private static final Logger LOG = LoggerFactory.getLogger(PrepareSoupEdgesBackwardsTask.class);

    private YPath todayEdgesBackwards;
    private YPath todayEdges;

    /* Due to CRYPTR-664 table with backward edges must be dynamic and and schema must contain
    compound key of id1Type, id1, id2Type, id2, sourceType, logSource in strict order
    */
    private YTreeNode getBackwardsEdgesSchema() {
        YTreeBuilder listBuilder = YTree.builder().beginAttributes()
                .key("strict").value(true)
                .key("unique_keys").value(true)
                .endAttributes()
                .beginList();

        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("id1Type"))
                .key("type").value(YTree.stringNode("string"))
                .key("sort_order").value(YTree.stringNode("ascending")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("id1"))
                .key("type").value(YTree.stringNode("string"))
                .key("sort_order").value(YTree.stringNode("ascending")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("id2Type"))
                .key("type").value(YTree.stringNode("string"))
                .key("sort_order").value(YTree.stringNode("ascending")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("id2"))
                .key("type").value(YTree.stringNode("string"))
                .key("sort_order").value(YTree.stringNode("ascending")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("sourceType"))
                .key("type").value(YTree.stringNode("string"))
                .key("sort_order").value(YTree.stringNode("ascending")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("logSource"))
                .key("type").value(YTree.stringNode("string"))
                .key("sort_order").value(YTree.stringNode("ascending")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("dates"))
                .key("type").value(YTree.stringNode("any")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("datesWeight"))
                .key("type").value(YTree.stringNode("double")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("indevice"))
                .key("type").value(YTree.stringNode("boolean")).endMap().build().mapNode());
        listBuilder.value(YTree.mapBuilder()
                .key("name").value(YTree.stringNode("survivalWeight"))
                .key("type").value(YTree.stringNode("double")).endMap().build().mapNode());

        return listBuilder.endList().build().listNode();
    }


    public PrepareSoupEdgesBackwardsTask(Dao dao, YPath workdir, SoupDynamicProcessingParams matchingParams) {
        super(dao, workdir, matchingParams);
        this.todayEdgesBackwards = workdir.child("soup_edges_backwards");
        this.todayEdges = workdir.child("soup_edges");
    }

    @Override
    public void runImpl(EmptyInput soupDir) {
        ListF<YPath> inputTables = Cf.list(todayEdges);

        dao.ytCypress().ensureDir(workdir);

        dao.ytTr().withTransactionAndTmpDir((tx, tmpDir) -> {

            LOG.info("Mapping soup edges to backwards edges");
            Option<GUID> txId = Option.of(tx.getId());

            PrepareSoupEdgesBackwards mapper = new PrepareSoupEdgesBackwards();

            YPath todayEdgesBackwardsTmp1 = dao.ytCypress().createTableWithSchema(
                    txId, tmpDir.child("soup_edges_backwards_tmp1"), Edge.class
            );

            MapOperation op = dao.ytOps().mapOperation(
                    inputTables,
                    Cf.list(todayEdgesBackwardsTmp1),
                    mapper
            );
            // double size of output causes job slowdown
            // make jobs smaller
            op.getMapSpecBuilder().setDataSizePerJob(DataSize.fromGigaBytes(1));
            op.runSync(txId);

            YPath todayEdgesBackwardsTmp2 = dao.ytCypress().createTableWithSchemaForDynamic(
                    txId, tmpDir.child("soup_edges_backwards_tmp2"), getBackwardsEdgesSchema()
            );

            Await.all(
                    dao.ytOps().sortAsync(
                            txId, Cf.list(todayEdgesBackwardsTmp1), todayEdgesBackwardsTmp2,
                            Cf.list("id1Type", "id1", "id2Type", "id2", "sourceType", "logSource")
                    )
            );

            dao.ytCypress().move(txId, todayEdgesBackwardsTmp2, todayEdgesBackwards);
            dao.ytCypress().remove(txId, todayEdgesBackwardsTmp1);
            dao.ytCypress().remove(txId, todayEdgesBackwardsTmp2);


        });

        dao.yt().tables().alterTable(todayEdgesBackwards, Optional.of(true), Optional.of(getBackwardsEdgesSchema()));
        dao.yt().tables().mount(todayEdgesBackwards);
    }

    @Override
    public YPath getOutput() {
        if (params.getSoupTable().isPresent()) {
            return params.getSoupTable().get();
        } else {
            return todayEdgesBackwards;
        }
    }

    @Override
    public String getDescription() {
        return "Prepares dynamic backwards edges table from static one";
    }
}
