package ru.yandex.crypta.graph2.soup.dynamic.workflow;

import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.model.soup.props.VertexPropertiesSchemaHelper;
import ru.yandex.crypta.graph2.soup.dynamic.config.SoupDynamicProcessingParams;
import ru.yandex.crypta.graph2.workflow.EmptyInput;
import ru.yandex.crypta.graph2.workflow.Task;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.operations.utils.LambdaReducerWithKey;
import ru.yandex.inside.yt.kosher.impl.operations.utils.functions.YtFunction;
import ru.yandex.inside.yt.kosher.impl.operations.utils.functions.YtFunction4V;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yt.ytclient.tables.TableSchema;

// import ru.yandex.inside.yt.kosher.impl.operations.utils.LambdaReducer;

public class PrepareVerticesPropertiesDynamicTask extends Task<EmptyInput, YPath, SoupDynamicProcessingParams> {
    private static final Logger LOG = LoggerFactory.getLogger(PrepareVerticesPropertiesDynamicTask.class);

    private YPath verticesPropertiesMerged;
    private YPath verticesPropertiesMergedDynamic;

    public PrepareVerticesPropertiesDynamicTask(Dao dao, YPath workdir, SoupDynamicProcessingParams params) {
        super(dao, workdir, params);

        this.verticesPropertiesMerged = workdir.child("vertices_properties");
        this.verticesPropertiesMergedDynamic = workdir.child("vertices_properties_dynamic");
    }

    @Override
    public void runImpl(EmptyInput emptyInput) {
        dao.ytCypress().ensureDir(workdir);

        if (!dao.ytCypress().isEmpty(verticesPropertiesMerged)) {

            TableSchema finalSchema = VertexPropertiesSchemaHelper
                    .getUnionVertexPropertiesSchema()
                    .toBuilder()
                    .setStrict(true)
                    .setUniqueKeys(true)
                    .sortBy(List.of("id_type", "id", "socdem_source", "source"))
                    .build();

            dao.ytTr().withTransactionAndTmpDir((tr, tmpDir) -> {
                Option<GUID> txId = Option.of(tr.getId());
                LOG.info("Preparing dynamic table for vertices properties...");
                YPath tmpVerticesTable = tmpDir.child("vertices");

                dao.ytCypress().createTableWithSchema(
                        txId,
                        verticesPropertiesMergedDynamic,
                        finalSchema
                );

                dao.ytOps().mapReduceOperation(
                    Cf.list(verticesPropertiesMerged.withColumns(Cf.wrap(finalSchema.getColumnNames()))),  // in
                    Cf.list(tmpVerticesTable),  // out
                    null,  // mapper
                    Cf.list("id_type", "id", "socdem_source", "source"),  // extended key
                    // reducer
                        new LambdaReducerWithKey<>(
                            YTableEntryTypes.YSON, YTableEntryTypes.YSON,
                            (YtFunction<YTreeMapNode, Tuple4<String, String, String, String>>) entry -> {
                                return
                                    new Tuple4<>(
                                        entry.getStringO("id_type").orElse(""),
                                        entry.getStringO("id").orElse(""),
                                        entry.getStringO("socdem_source").orElse(""),
                                        entry.getStringO("source").orElse("")
                                    )
                                ;
                            },
                            (YtFunction4V<Tuple4<String, String, String, String>, Iterator<YTreeMapNode>, Yield<YTreeMapNode>, Statistics>)
                            (key, entry, yield, statistics) -> {
                                yield.yield(entry.next());
                            }
                        )
                )
                    .runSync(txId);

                dao.ytOps().sortOperation(
                        Cf.list(tmpVerticesTable),  // in
                        verticesPropertiesMergedDynamic,
                        Cf.list("id_type", "id", "socdem_source", "source")
                ).withSchemaInferenceMode("from_output")
                        .runSync(txId);
            });

            dao.yt().tables().alterTable(
                    verticesPropertiesMergedDynamic,
                    Optional.of(true),
                    Optional.of(finalSchema.toYTree())
            );

            dao.yt().tables().mount(verticesPropertiesMergedDynamic);

            LOG.info("Preparing dynamic table for vertices properties... DONE");
        } else {
            LOG.info("Vertices properties table is empty");
        }
    }

    @Override
    public YPath getOutput() {
        if (params.getSoupTable().isPresent()) {
            return params.getIdsStorageTable().get();
        } else {
            return verticesPropertiesMergedDynamic;
        }
    }

    @Override
    public String getDescription() {
        return "Prepares dynamic vertices table from static one";
    }
}
