import logging

from crypta.lib.python.bt.workflow import IndependentTask
from crypta.lib.python.yql_runner.task import YQLRunnerTask
from crypta.lib.python.bt.tasks import (
    YtTask,
)

logger = logging.getLogger(__name__)


class Paths(object):
    SOUP_EDGES = "//home/crypta/production/state/graph/v2/soup/cooked/soup_edges"
    EDGE_MESSAGES = "//home/crypta/production/state/graph/v2/soup/cooked/edge_messages"

    SOUP_EDGE_EXP = "//home/crypta/production/state/graph/v2/soup/cooked/soup_edges_exp"
    EDGE_MESSAGES_EXP = "//home/crypta/production/state/graph/v2/soup/cooked/edge_messages_exp"

    MAIN_ATTRIBUTES = ["run_date", "generate_date"]

    @staticmethod
    def path_to_attr(path, attr_name):
        return "{}/@{}".format(path, attr_name)

    @staticmethod
    def share_main_attrs(yt, path_from, path_to):
        for attr in Paths.MAIN_ATTRIBUTES:
            attr_path = Paths.path_to_attr(path_from, attr)
            if yt.exists(attr_path):
                value = yt.get(attr_path)
                yt.set(Paths.path_to_attr(path_to, attr), value)

    @staticmethod
    def check_if_main_attrs_equal(yt, path_first, path_second):
        for attr in Paths.MAIN_ATTRIBUTES:
            attr_path_first = Paths.path_to_attr(path_first, attr)
            attr_path_second = Paths.path_to_attr(path_second, attr)
            if not yt.exists(attr_path_first) or not yt.exists(attr_path_second):
                return False
            if yt.get(attr_path_first) != yt.get(attr_path_second):
                return False
        return True


class PrepareExpSoupFromQuery(YQLRunnerTask, IndependentTask):
    @property
    def query_template(self):
        return "prepare_exp_soup.sql.j2"

    @property
    def input_soup(self):
        return Paths.SOUP_EDGES

    @property
    def output_soup(self):
        return Paths.SOUP_EDGE_EXP

    def get_context_data(self, **kwargs):
        context = super(PrepareExpSoupFromQuery, self).get_context_data(**kwargs)

        context.update(
            input_soup=self.input_soup,
            output_exp_soup=self.output_soup
        )
        return context

    def run(self, *args, **kwargs):
        for path in [Paths.SOUP_EDGE_EXP, Paths.EDGE_MESSAGES_EXP]:
            if self.yt.exists(path):
                self.yt.remove(path)

        result = super(PrepareExpSoupFromQuery, self).run(*args, **kwargs)
        logger.info("Result:")
        logger.info(result)
        Paths.share_main_attrs(self.yt, self.input_soup, self.output_soup)

        # Human Matching uses edge_messages if it exists
        # in the other case edge_messages will be created from soup_edges
        # so we can recreate soup_edges only


class MakeLinksToProdTables(YtTask, IndependentTask):
    def make_link(self, input_path, output_path):
        if self.yt.exists(output_path):
            self.yt.remove(output_path)
        self.yt.link(input_path, output_path, recursive=True, ignore_existing=True)

    def run(self, *args, **kwargs):
        for path, path_exp in [
            (Paths.SOUP_EDGES, Paths.SOUP_EDGE_EXP),
            (Paths.EDGE_MESSAGES, Paths.EDGE_MESSAGES_EXP)
        ]:
            self.make_link(path, path_exp)


class PrepareExpSoup(YtTask, IndependentTask):
    def run(self, *args, **kwargs):
        # yield MakeLinksToProdTables()  # default
        yield PrepareExpSoupFromQuery()  # exp CRYPTA-9337
