# -*- coding: utf-8 -*-
from textwrap import dedent

from crypta.graph.metrics.lib import YqlDailyReportBase, SolomonSensorConfig
from crypta.graph.soup.config.python import ID_TYPE
from crypta.graph.soupy_indevice.statface.lib.common import get_table, SolomonMixin


class DistMetricsReport(SolomonMixin, YqlDailyReportBase):

    report_name = 'Crypta/Graph/Indevice/DistMetrics'
    indevice_root = '//home/crypta/production/state/graph/v2/indevice'
    idstorage_root = '//home/crypta/production/ids_storage'
    kind = 'default'

    def idstorage(self, idt):
        return '{}/{}/eternal'.format(self.idstorage_root, idt.Name)

    @property
    def query(self):
        dt, indevice_table = get_table(self.indevice_root, self.report_date)

        yql = '''
            PRAGMA SimpleColumns;

            $dt2ts = ($dt) -> {{
                RETURN DateTime::ToSeconds(CAST($dt AS Date)) ?? 0L;
            }};

            $threshold_intersection = 24 * 60 * 60;
            -- UDAF to find if there's an intersection longer than 24h among all the pairwise intersections of id activity intervals
            -- This maintains the largest possible intersection interval (a union of intervals seen so far).
            -- Assumes that the list of intervals is sorted by starting point.

            -- Initial state: one open interval (the current one), no intersection
            $create = ($i, $parent) -> {{ return ($i, false); }};

            -- Add a new interval: try to intersect an incoming interval with all the open ones,
            -- remove open intervals that are too far to the left.
            $add = ($state, $i, $parent) -> {{
                $int_left = nvl(ListMax(AsList($i.0, ($state.0).0)), 0);
                $int_right = nvl(ListMin(AsList($i.1, ($state.0).1)), 0);
                $max_intersection_length = if($int_left <= $int_right, $int_right - $int_left, 0);

                $new_interval = (($state.0).0, nvl(ListMax(AsList($i.1, ($state.0).1)), 0));

                return
                    if($state.1, -- stop doing anything if a suitable intersection is found
                       $state,
                       if($max_intersection_length > 0,
                          ($new_interval, $max_intersection_length >= $threshold_intersection),
                          ($i, false)
                       )
                    );
            }};

            -- Merge two states (probably isn't used?)
            $merge = ($state1, $state2) -> {{
                return (
                    ListExtend($state1.0, $state2.0),
                    nvl(ListMax(AsList($state1.1, $state2.1)), 0)
                );
            }};

            $indevice_total = (
                SELECT COUNT(DISTINCT indevice_id)
                FROM `{indevice}`
            );
        '''.format(indevice=indevice_table)

        for idtype in [ID_TYPE.YANDEXUID, ID_TYPE.ICOOKIE]:
            yql += '''
                $joined_{idt} = (
                    select a.indevice_id as indevice_id,
                           a.id as id,
                           a.id_type as id_type,
                           ($dt2ts(a.date_begin), $dt2ts(a.date_end)) as ts_interval,
                           b.browser_name as browser_name
                    from `{indevice}` as a
                    inner join `{idstorage}` as b
                    on a.id = b.id and a.id_type = b.id_type
                    where b.is_browser
                );

                $intersections_{idt} = (
                    select UDAF(nvl(ts_interval, (0, 0)), $create, $add, $merge) over w as intersections,
                        indevice_id, id, id_type, browser_name
                    from $joined_{idt}
                    window w as (
                        PARTITION BY indevice_id, id_type, browser_name
                        order by ts_interval.0
                    )
                );
            '''.format(idt=idtype.Name, indevice=indevice_table, idstorage=self.idstorage(idtype))

        for idtype in [ID_TYPE.IDFA, ID_TYPE.GAID, ID_TYPE.MM_DEVICE_ID]:
            yql += '''
                $joined_{idt} = (
                    select a.indevice_id as indevice_id,
                           a.id as id,
                           a.id_type as id_type,
                           ($dt2ts(a.date_begin), $dt2ts(a.date_end)) as ts_interval
                    from `{indevice}` as a
                    inner join `{idstorage}` as b
                    on a.id = b.id and a.id_type = b.id_type
                );

                $intersections_{idt} = (
                    select UDAF(nvl(ts_interval, (0, 0)), $create, $add, $merge) over w as intersections,
                        indevice_id, id, id_type
                    from $joined_{idt}
                    WINDOW w AS (
                        PARTITION BY indevice_id, id_type
                        ORDER BY ts_interval.0
                    )
                );
            '''.format(idt=idtype.Name, indevice=indevice_table, idstorage=self.idstorage(idtype))

        yql += '''
            $joined_{idt} = (
                select a.indevice_id as indevice_id,
                       a.id as id,
                       a.id_type as id_type,
                       ($dt2ts(a.date_begin), $dt2ts(a.date_end)) as ts_interval,
                       ListSortDesc(DictItems(
                            Yson::ConvertToInt64Dict(b.api_keys)),
                            ($x) -> {{ return $x.1; }}
                        )[0].0 AS main_api_key  -- apikey with most hits
                FROM `{indevice}` AS a
                INNER JOIN `{idstorage}` AS b
                on a.id = b.id and a.id_type = b.id_type
            );

            $intersections_{idt} = (
                select UDAF(nvl(ts_interval, (0, 0)), $create, $add, $merge) over w as intersections,
                    indevice_id, id, id_type, main_api_key
                from $joined_{idt}
                window w as (
                    PARTITION BY indevice_id, id_type, main_api_key
                    order by ts_interval.0
                )
            );
        '''.format(idt=ID_TYPE.UUID.Name, indevice=indevice_table, idstorage=self.idstorage(ID_TYPE.UUID))

        yql += '''
            $combined_traits = (
                SELECT * FROM (
                    SELECT id, id_type, manufacturer, model, screen_width, screen_height
                    FROM `{idstorage_idfa}`
                    UNION ALL
                    SELECT id, id_type, manufacturer, model, screen_width, screen_height
                    FROM `{idstorage_gaid}`
                    UNION ALL
                    SELECT id, id_type, manufacturer, model, screen_width, screen_height
                    FROM `{idstorage_mm_devid}`
                ) WHERE (manufacturer ?? model) IS NOT NULL
                    AND (screen_width ?? screen_height) IS NOT NULL
            );

            $indevice_traits = (
                select a.indevice_id as indevice_id,
                       b.manufacturer as manufacturer,
                       b.model as model,
                       b.screen_width as screen_width,
                       b.screen_height as screen_height
                from `{indevice}` as a
                inner join $combined_traits as b
                on a.id = b.id and a.id_type = b.id_type
            );

            $distinct_models = (
                select distinct indevice_id, manufacturer, model
                from $indevice_traits
            );

            $inconsistent_models = (
                select indevice_id, count(*)
                from $distinct_models
                group by indevice_id
                having count(*) > 1
            );

            $distinct_screensizes = (
                select distinct indevice_id, screen_width, screen_height
                from $indevice_traits
            );

            $inconsistent_screensizes = (
                select indevice_id, count(*)
                from $distinct_screensizes
                group by indevice_id
                having count(*) > 1
            );

            $inconsistent_models_count = (select count(*) from $inconsistent_models);
            $inconsistent_screensizes_count = (select count(*) from $inconsistent_screensizes);
        '''.format(
            idstorage_idfa=self.idstorage(ID_TYPE.IDFA),
            idstorage_gaid=self.idstorage(ID_TYPE.GAID),
            idstorage_mm_devid=self.idstorage(ID_TYPE.MM_DEVICE_ID),
            indevice=indevice_table)

        final_selects = []
        for idtype in [ID_TYPE.YANDEXUID, ID_TYPE.ICOOKIE, ID_TYPE.IDFA, ID_TYPE.GAID, ID_TYPE.MM_DEVICE_ID, ID_TYPE.UUID]:
            yql += '''
                $indevice_conflicting_{idt} = (
                    select count(distinct(indevice_id))
                    FROM $intersections_{idt}
                    WHERE intersections.1
                );

            '''.format(idt=idtype.Name)

            final_selects.append('''
                CAST($indevice_conflicting_{idt} AS Double) / $indevice_total * 100 AS conflicting_{idt}_fraction
            '''.format(idt=idtype.Name))

        final_selects.append('''
                CAST($inconsistent_models_count AS Double) / $indevice_total * 100 AS inconsistent_models_fraction
            ''')
        final_selects.append('''
                CAST($inconsistent_screensizes_count AS Double) / $indevice_total * 100 AS inconsistent_screensizes_fraction
            ''')

        yql += '''
            SELECT '{date}' as fielddate,
                   '{kind}' as kind,
        '''.format(date=dt, kind=self.kind)
        yql += ',\n'.join(final_selects)

        return dedent(yql)

    report_config = dedent(u'''
        ---
        dimensions:
            - fielddate: date
            - kind: string
        measures:
            - conflicting_icookie_fraction: number
            - conflicting_yandexuid_fraction: number
            - conflicting_idfa_fraction: number
            - conflicting_mm_device_id_fraction: number
            - conflicting_uuid_fraction: number
            - conflicting_gaid_fraction: number
            - inconsistent_screensizes_fraction: number
            - inconsistent_models_fraction: number
        graphs:
            - title: 'Доля indevice_id с конфликтующими куками'
              fields:
                - conflicting_icookie_fraction
                - conflicting_yandexuid_fraction
              titles:
                conflicting_icookie_fraction: 'icookie'
                conflicting_yandexuid_fraction: 'yandexuid'
            - title: 'Доля indevice_id с конфликтующими идентификаторами устройств и приложений'
              fields:
                - conflicting_idfa_fraction
                - conflicting_mm_device_id_fraction
                - conflicting_gaid_fraction
                - conflicting_uuid_fraction
              titles:
                conflicting_idfa_fraction: 'IDFA'
                conflicting_mm_device_id_fraction: 'mm_device_id'
                conflicting_gaid_fraction: 'GAID'
                conflicting_uuid_fraction: 'UUID'
            - title: 'Доля indevice_id с конфликтами моделей и размеров экранов'
              fields:
                - inconsistent_screensizes_fraction
                - inconsistent_models_fraction
              titles:
                inconsistent_screensizes_fraction: 'Размер экрана'
                inconsistent_models_fraction: 'Модель'
    ''')

    solomon_sensors = SolomonSensorConfig(
        sensor_keys=[
            'conflicting_icookie_fraction',
            'conflicting_yandexuid_fraction',
            'conflicting_idfa_fraction',
            'conflicting_mm_device_id_fraction',
            'conflicting_uuid_fraction',
            'conflicting_gaid_fraction',
            'inconsistent_screensizes_fraction',
            'inconsistent_models_fraction'
        ],
        labels=['kind']
    )
