# -*- coding: utf-8 -*-
from textwrap import dedent

from crypta.graph.metrics.lib import YqlDailyReportBase, SolomonSensorConfig
from crypta.graph.soupy_indevice.statface.lib.common import get_table, SolomonMixin


class IdCountReport(SolomonMixin, YqlDailyReportBase):
    report_name = 'Crypta/Graph/Indevice/IdCounts'
    indevice_root = '//home/crypta/production/state/graph/v2/indevice'
    kind = 'default'

    @property
    def query(self):
        dt_last, last = get_table(self.indevice_root, self.report_date, 0)

        return dedent('''
            $indevice = '{indevice}';
            $ancient_history = '2015-10-01';

            $counts = (
                select indevice_id, id_type, cluster_date_begin, count(*) as c
                from $indevice
                group by indevice_id, id_type, cluster_date_begin
            );

            $count_perc = (
                select
                    id_type,
                    (cluster_date_begin ?? $ancient_history) as cluster_date_begin,
                    max(c) as p100,
                    percentile(c, 0.999) as p999,
                    percentile(c, 0.99) as p99,
                    percentile(c, 0.95) as p95,
                    percentile(c, 0.75) as p75,
                    percentile(c, 0.50) as p50
                from $counts
                group by id_type, cluster_date_begin
            );

            select
                cluster_date_begin as fielddate,
                '{kind}' as kind,
                id_type,
                '{date}' as table_date,
                p100, p999, p99, p95, p75, p50
            from $count_perc
        '''.format(
            indevice=last,
            date=dt_last,
            kind=self.kind
        ))

    report_config = dedent('''
        ---
        dimensions:
            - fielddate: date
            - kind: string
            - id_type: string
            - table_date: string
        measures:
            - p100: number
            - p999: number
            - p99: number
            - p95: number
            - p75: number
            - p50: number
    ''')

    solomon_sensors = SolomonSensorConfig(
        sensor_keys=['p100', 'p999', 'p99', 'p95', 'p75', 'p50'],
        labels=['kind', 'id_type', 'table_date']
    )
