# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
from textwrap import dedent

import yt.wrapper as yt

from crypta.graph.metrics.lib import YqlDailyReportBase, SolomonSensorConfig
from crypta.graph.soupy_indevice.statface.lib.common import get_table, SolomonMixin


class SizesAndTrafficReport(SolomonMixin, YqlDailyReportBase):

    report_name = "Crypta/Graph/Indevice/SizeTraf"
    indevice_root = "//home/crypta/production/state/graph/v2/indevice"
    kind = "default"

    @property
    def query(self):
        dt_indevice, indevice_table = get_table(self.indevice_root, self.report_date)
        _, devid_dayuse = get_table("//statbox/cube/total/daily/dayuse/mobile/v4", "last")
        _, yuid_traf_table = get_table("//statbox/hypercube/total/user_history/source/traffic_history/v2", "last")

        sizes_table = indevice_table + "_sizes"
        bad_ids_table = indevice_table + "_bad_ids"

        if not yt.exists(sizes_table):
            sizes_table = "{}/sizes/{}".format(self.indevice_root, dt_indevice)

        if not yt.exists(bad_ids_table):
            bad_ids_table = "{}/bad_ids/{}".format(self.indevice_root, dt_indevice)

        date_from = datetime.strptime(dt_indevice, "%Y-%m-%d") - timedelta(days=30)
        return dedent(
            """
            pragma SimpleColumns;
            pragma yt.InferSchema;

            $date = '{date}';
            $date_from = '{date_from}';

            $devid_dayuse = '{devid_dayuse}';
            $yuid_traffic_table = '{yuid_traffic_table}';
            $indevice = '{indevice_table}';
            $sizes_table = '{sizes_table}';
            $bad_ids_table = '{bad_ids_table}';

            $normalize_devid = ($x) -> {{
                return String::ToLower(String::ReplaceAll($x, '-', ''));
            }};

            -- matching data

            $devids = (
                select indevice_id, $normalize_devid(id) as id
                from $indevice
                where id_type = 'mm_device_id'

                union all

                select 'bad-ids' as indevice_id, $normalize_devid(id) as id
                from $bad_ids_table
                where id_type = 'mm_device_id'
            );

            $distinct_devids = (
                select distinct indevice_id, id
                from $devids
            );

            $yuids = (
                select indevice_id, id
                from $indevice
                where id_type = 'yandexuid'

                union all

                select 'bad-ids' as indevice_id, id
                from $bad_ids_table
                where id_type = 'yandexuid'
            );

            $distinct_yuids = (
                select distinct indevice_id, id
                from $yuids
            );

            $size_percentiles = (
                select
                    AsDict(
                        AsTuple('p50', cast(Math::Ceil(percentile(size, 0.50)) as uint64)),
                        AsTuple('p75', cast(Math::Ceil(percentile(size, 0.75)) as uint64)),
                        AsTuple('p90', cast(Math::Ceil(percentile(size, 0.90)) as uint64)),
                        AsTuple('p99', cast(Math::Ceil(percentile(size, 0.99)) as uint64)),
                        AsTuple('p999', cast(Math::Ceil(percentile(size, 0.999)) as uint64)),
                        AsTuple('p9999', cast(Math::Ceil(percentile(size, 0.9999)) as uint64)),
                        AsTuple('p100_no_max', cast(TOP(size, 2)[1] as uint64)),
                        AsTuple('p100', cast(max(size) as uint64))
                    )
                from $sizes_table
            );

            -- etalon data
            -- by devid
            $devid_dates = (
                select $normalize_devid(device_id) as device_id
                from $devid_dayuse
                where (`date` between $date_from and $date)
            );

            $devid_count_total = (select count(*) from $devid_dates);

            $devid_traffic_by_indevice = (
                select
                    s.indevice_id as indevice_id,
                    count(*) as visit_abs
                from $devid_dates as hc
                inner join $distinct_devids as s on hc.device_id == s.id
                group by s.indevice_id
            );

            $devid_traffic = (
                select
                    a.indevice_id as indevice_id,
                    a.visit_abs as visit_abs,
                    case when a.indevice_id = 'bad-ids' then 'bad'
                         when b.size <= $size_percentiles['p50'] then 'p50'
                         when b.size <= $size_percentiles['p75'] then 'p75'
                         when b.size <= $size_percentiles['p90'] then 'p90'
                         when b.size <= $size_percentiles['p99'] then 'p99'
                         when b.size <= $size_percentiles['p999'] then 'p999'
                         when b.size <= $size_percentiles['p9999'] then 'p9999'
                         when b.size <= $size_percentiles['p100_no_max'] then 'p100_no_max'
                         else 'p100'
                    end as perc
                from $devid_traffic_by_indevice as a
                left join $sizes_table as b using (indevice_id)
            );

            $devid_traffic_by_perc = (
                select ToDict(tups) from (
                    select AGGREGATE_LIST(tup) as tups from (
                        select AsTuple(perc, sum(visit_abs)) as tup
                        from $devid_traffic
                        group by perc
                    )
                )
            );

            -- by yuid
            $yuid_traffic_dates = (
                select yandexuid, visits_count, yandex_visits_count
                from $yuid_traffic_table as a
                where (`date` between $date_from and $date) and yandexuid is not null
            );

            $yuid_visit_count_total = (select sum(visits_count) from $yuid_traffic_dates);
            $yuid_yandex_visit_count_total = (select sum(yandex_visits_count) from $yuid_traffic_dates);

            $yuid_traffic_by_indevice = (
                select b.indevice_id as indevice_id,
                       sum(a.visits_count) as visits,
                       sum(a.yandex_visits_count) as yandex_visits
                from $yuid_traffic_dates as a
                inner join $distinct_yuids as b on a.yandexuid = b.id
                group by b.indevice_id
            );

            $yuid_traffic = (
                select
                    a.indevice_id as indevice_id,
                    a.visits as visits,
                    a.yandex_visits as yandex_visits,
                    b.size as size,
                    case when a.indevice_id = 'bad-ids' then 'bad'
                         when b.size <= $size_percentiles['p50'] then 'p50'
                         when b.size <= $size_percentiles['p75'] then 'p75'
                         when b.size <= $size_percentiles['p90'] then 'p90'
                         when b.size <= $size_percentiles['p99'] then 'p99'
                         when b.size <= $size_percentiles['p999'] then 'p999'
                         when b.size <= $size_percentiles['p9999'] then 'p9999'
                         when b.size <= $size_percentiles['p100_no_max'] then 'p100_no_max'
                         else 'p100'
                    end as perc
                from $yuid_traffic_by_indevice as a
                left join $sizes_table as b using (indevice_id)
            );

            $yuid_traffic_by_perc = (
                select ToDict(tups) from (
                    select AGGREGATE_LIST(tup) as tups from (
                        select AsTuple(perc, AsTuple(sum(visits), sum(yandex_visits))) as tup
                        from $yuid_traffic
                        group by perc
                    )
                )
            );

            -- resulting metric

            select
                '{date}' as fielddate,
                '{kind}' as kind,
                $size_percentiles['p50'] as size_p50,
                $size_percentiles['p75'] as size_p75,
                $size_percentiles['p90'] as size_p90,
                $size_percentiles['p99'] as size_p99,
                $size_percentiles['p999'] as size_p999,
                $size_percentiles['p9999'] as size_p9999,
                $size_percentiles['p100_no_max'] as size_p100_no_max,
                $size_percentiles['p100'] as size_p100,

                $devid_traffic_by_perc['p50'] ?? 0 as dev_traf_p50,
                $devid_traffic_by_perc['p75'] ?? 0 as dev_traf_p75,
                $devid_traffic_by_perc['p90'] ?? 0 as dev_traf_p90,
                $devid_traffic_by_perc['p99'] ?? 0 as dev_traf_p99,
                $devid_traffic_by_perc['p999'] ?? 0 as dev_traf_p999,
                $devid_traffic_by_perc['p9999'] ?? 0 as dev_traf_p9999,
                $devid_traffic_by_perc['p100_no_max'] ?? 0 as dev_traf_p100_no_max,
                $devid_traffic_by_perc['p100'] ?? 0 as dev_traf_p100,
                $devid_traffic_by_perc['bad'] ?? 0 as dev_traf_bad,
                $devid_count_total as dev_traf_total,

                $yuid_traffic_by_perc['p50'].0 ?? 0 as yuid_traf_p50,
                $yuid_traffic_by_perc['p75'].0 ?? 0 as yuid_traf_p75,
                $yuid_traffic_by_perc['p90'].0 ?? 0 as yuid_traf_p90,
                $yuid_traffic_by_perc['p99'].0 ?? 0 as yuid_traf_p99,
                $yuid_traffic_by_perc['p999'].0 ?? 0 as yuid_traf_p999,
                $yuid_traffic_by_perc['p9999'].0 ?? 0 as yuid_traf_p9999,
                $yuid_traffic_by_perc['p100_no_max'].0 ?? 0 as yuid_traf_p100_no_max,
                $yuid_traffic_by_perc['p100'].0 ?? 0 as yuid_traf_p100,
                $yuid_traffic_by_perc['bad'].0 ?? 0 as yuid_traf_bad,
                $yuid_visit_count_total as yuid_traf_total,

                $yuid_traffic_by_perc['p50'].1 ?? 0 as yuid_ya_traf_p50,
                $yuid_traffic_by_perc['p75'].1 ?? 0 as yuid_ya_traf_p75,
                $yuid_traffic_by_perc['p90'].1 ?? 0 as yuid_ya_traf_p90,
                $yuid_traffic_by_perc['p99'].1 ?? 0 as yuid_ya_traf_p99,
                $yuid_traffic_by_perc['p999'].1 ?? 0 as yuid_ya_traf_p999,
                $yuid_traffic_by_perc['p9999'].1 ?? 0 as yuid_ya_traf_p9999,
                $yuid_traffic_by_perc['p100_no_max'].1 ?? 0 as yuid_ya_traf_p100_no_max,
                $yuid_traffic_by_perc['p100'].1 ?? 0 as yuid_ya_traf_p100,
                $yuid_traffic_by_perc['bad'].1 ?? 0 as yuid_ya_traf_bad,
                $yuid_yandex_visit_count_total as yuid_ya_traf_total
        """
        ).format(
            yuid_traffic_table=yuid_traf_table,
            devid_dayuse=devid_dayuse,
            indevice_table=indevice_table,
            sizes_table=sizes_table,
            bad_ids_table=bad_ids_table,
            date=dt_indevice,
            date_from=date_from.strftime("%Y-%m-%d"),
            kind=self.kind,
        )

    @property
    def report_config(self):
        return dedent(
            u"""
            ---
            dimensions:
                - fielddate: date
                - kind: string
            measures:
                - size_p50: number
                - size_p75: number
                - size_p90: number
                - size_p99: number
                - size_p999: number
                - size_p9999: number
                - size_p100_no_max: number
                - size_p100: number
                - dev_traf_p50: number
                - dev_traf_p75: number
                - dev_traf_p90: number
                - dev_traf_p99: number
                - dev_traf_p999: number
                - dev_traf_p9999: number
                - dev_traf_p100_no_max: number
                - dev_traf_p100: number
                - dev_traf_bad: number
                - dev_traf_total: number
                - yuid_traf_p50: number
                - yuid_traf_p75: number
                - yuid_traf_p90: number
                - yuid_traf_p99: number
                - yuid_traf_p999: number
                - yuid_traf_p9999: number
                - yuid_traf_p100_no_max: number
                - yuid_traf_p100: number
                - yuid_traf_bad: number
                - yuid_traf_total: number
                - yuid_ya_traf_p50: number
                - yuid_ya_traf_p75: number
                - yuid_ya_traf_p90: number
                - yuid_ya_traf_p99: number
                - yuid_ya_traf_p999: number
                - yuid_ya_traf_p9999: number
                - yuid_ya_traf_p100_no_max: number
                - yuid_ya_traf_p100: number
                - yuid_ya_traf_bad: number
                - yuid_ya_traf_total: number
            calculations:
                # TODO: remove dev_traf later CRYPTR-1794
                - dev_traf_unaccounted:
                    expression: "dev_traf_total - dev_traf_p50 - dev_traf_p75 - \
dev_traf_p90 - dev_traf_p99 - dev_traf_p999 - dev_traf_p9999 - dev_traf_p100_no_max - dev_traf_p100 - dev_traf_bad"
                - yuid_traf_unaccounted:
                    expression: "yuid_traf_total - yuid_traf_p50 - yuid_traf_p75 - \
yuid_traf_p90 - yuid_traf_p99 - yuid_traf_p999 - yuid_traf_p9999 - yuid_traf_p100_no_max - yuid_traf_p100 - yuid_traf_bad"
                - yuid_ya_traf_unaccounted:
                    expression: "yuid_ya_traf_total - yuid_ya_traf_p50 - yuid_ya_traf_p75 - \
yuid_ya_traf_p90 - yuid_ya_traf_p99 - yuid_ya_traf_p999 - yuid_ya_traf_p9999 - yuid_ya_traf_p100_no_max - \
yuid_ya_traf_p100 - yuid_ya_traf_bad"
            graphs:
                - title: 'Перцентили размеров'
                  fields:
                    - size_p50
                    - size_p75
                    - size_p90
                    - size_p99
                    - size_p999
                    - size_p9999
                    - size_p100_no_max
                    - size_p100
                  fields_order:
                    - size_p100
                    - size_p100_no_max
                    - size_p9999
                    - size_p999
                    - size_p99
                    - size_p90
                    - size_p75
                    - size_p50
                # TODO: remove dev_traf later CRYPTR-1794
                - title: 'Распределение аппметричных DeviceID по перцентилям размеров indevice_id'
                  type: 'stacked_100p'
                  fields:
                    - dev_traf_p50
                    - dev_traf_p75
                    - dev_traf_p90
                    - dev_traf_p99
                    - dev_traf_p999
                    - dev_traf_p9999
                    - dev_traf_p100_no_max
                    - dev_traf_p100
                    - dev_traf_bad
                    - dev_traf_unaccounted
                  fields_order:
                    - dev_traf_unaccounted
                    - dev_traf_bad
                    - dev_traf_p100
                    - dev_traf_p100_no_max
                    - dev_traf_p9999
                    - dev_traf_p999
                    - dev_traf_p99
                    - dev_traf_p90
                    - dev_traf_p75
                    - dev_traf_p50
                - title: 'Распределение посещений yuid по перцентилям размеров'
                  type: 'stacked_100p'
                  fields:
                    - yuid_traf_p50
                    - yuid_traf_p75
                    - yuid_traf_p90
                    - yuid_traf_p99
                    - yuid_traf_p999
                    - yuid_traf_p9999
                    - yuid_traf_p100_no_max
                    - yuid_traf_p100
                    - yuid_traf_bad
                    - yuid_traf_unaccounted
                  fields_order:
                    - yuid_traf_unaccounted
                    - yuid_traf_bad
                    - yuid_traf_p100
                    - yuid_traf_p100_no_max
                    - yuid_traf_p9999
                    - yuid_traf_p999
                    - yuid_traf_p99
                    - yuid_traf_p90
                    - yuid_traf_p75
                    - yuid_traf_p50
                - title: 'Распределение посещений Яндекса yuid по перцентилям размеров'
                  type: 'stacked_100p'
                  fields:
                    - yuid_ya_traf_p50
                    - yuid_ya_traf_p75
                    - yuid_ya_traf_p90
                    - yuid_ya_traf_p99
                    - yuid_ya_traf_p999
                    - yuid_ya_traf_p9999
                    - yuid_ya_traf_p100_no_max
                    - yuid_ya_traf_p100
                    - yuid_ya_traf_bad
                    - yuid_ya_traf_unaccounted
                  fields_order:
                    - yuid_ya_traf_unaccounted
                    - yuid_ya_traf_bad
                    - yuid_ya_traf_p100
                    - yuid_ya_traf_p100_no_max
                    - yuid_ya_traf_p9999
                    - yuid_ya_traf_p999
                    - yuid_ya_traf_p99
                    - yuid_ya_traf_p90
                    - yuid_ya_traf_p75
                    - yuid_ya_traf_p50
    """
        )

    solomon_sensors = SolomonSensorConfig(
        sensor_keys=[
            "size_p50",
            "size_p75",
            "size_p90",
            "size_p99",
            "size_p999",
            "size_p9999",
            "size_p100_no_max",
            "size_p100",
            "yuid_traf_p50",
            "yuid_traf_p75",
            "yuid_traf_p90",
            "yuid_traf_p99",
            "yuid_traf_p999",
            "yuid_traf_p9999",
            "yuid_traf_p100_no_max",
            "yuid_traf_p100",
            "yuid_traf_bad",
            "yuid_traf_total",
            "yuid_ya_traf_p50",
            "yuid_ya_traf_p75",
            "yuid_ya_traf_p90",
            "yuid_ya_traf_p99",
            "yuid_ya_traf_p999",
            "yuid_ya_traf_p9999",
            "yuid_ya_traf_p100_no_max",
            "yuid_ya_traf_p100",
            "yuid_ya_traf_bad",
            "yuid_ya_traf_total",
        ],
        labels=["kind"],
    )
