# -*- coding: utf-8 -*-
from textwrap import dedent

from crypta.graph.metrics.lib import YqlDailyReportBase, SolomonSensorConfig
from crypta.graph.soup.config.python import ID_TYPE
from crypta.graph.soupy_indevice.statface.lib.common import get_table, SolomonMixin


class YandexAppsCoverage(SolomonMixin, YqlDailyReportBase):

    report_name = 'Crypta/Graph/Indevice/YandexAppsCoverage'
    indevice_root = '//home/crypta/production/state/graph/v2/indevice'
    idstorage_root = '//home/crypta/public/ids_storage'
    matching_root = '//home/crypta/public/matching/by_id'
    kind = 'default'

    def idstorage(self, idt):
        return '{}/{}/eternal'.format(self.idstorage_root, idt.Name)

    def soup_matching_dir(self, id1, id2):
        return '{}/{}/soup/{}'.format(self.matching_root, id1.Name, id2.Name)

    @property
    def query(self):
        dt_last, last = get_table(self.indevice_root, self.report_date)

        return dedent('''
            -- yandexuids

            $yandexuids =
            SELECT
                id,
                id_type,
                CASE
                    WHEN String::Contains(browser_name, 'YandexSearch') THEN 'pp'
                    WHEN String::Contains(browser_name, 'YandexBrowser') THEN 'yabro'
                    ELSE 'other'
                END as browser
            FROM `{idstorage_yandexuid}`;

            -- direct browsers by uuid

            $soup_apps_direct_yandexuid =
            select
                IF(id1Type == 'uuid', id1, id2) as id,
                'uuid' as id_type,
                IF(id1Type == 'yandexuid', id1, id2) as yandexuid
            from range('{soup_yandexuid_uuid}');

            $soup_apps_direct_browsers =
            select
                id,
                id_type,
                AGGREGATE_LIST_DISTINCT(y.browser) as browsers
            from $soup_apps_direct_yandexuid as soup
            inner join $yandexuids as y
            on (soup.yandexuid = y.id)
            group by
                soup.id as id,
                soup.id_type as id_type;

            -- matched browsers by indevice_id
            $browsers_by_indevice_id =
            select
                indevice_id,
                ListSort(AGGREGATE_LIST_DISTINCT(info.browser)) as browsers
            from $yandexuids as info
            inner join any `{indevice_last}` as indev
            using (id_type, id)
            group by indev.indevice_id as indevice_id;


            -- apps
            $yabro_pp_apps =
            SELECT
                id, id_type, os,
                case
                    when app_id == 'ru.yandex.searchplugin' then 'pp'
                    when app_id == 'ru.yandex.mobile.search' then 'yabro'
                    when app_id == 'ru.yandex.mobile' then 'pp'
                    when app_id == 'com.yandex.browser' then 'yabro'
                    else 'wtf'
                end as app
            FROM `{idstorage_uuid}`
            where app_id in (
                'ru.yandex.searchplugin',
                'ru.yandex.mobile.search',
                'ru.yandex.mobile',
                'com.yandex.browser'
            ) and os != '' and os != 'unknown';
            -- and date_begin > '2020-01-01';

            -- join apps to browsers

            $yabro_pp_apps_indevice_id =
            select
                info.os as os,
                info.app as app,
                indev.indevice_id as indevice_id,
                soup.browsers as direct_browsers
            from $yabro_pp_apps as info
            left join any `{indevice_last}` as indev
            on info.id_type == indev.id_type and info.id = indev.id
            left join any $soup_apps_direct_browsers as soup
            on info.id_type == soup.id_type and info.id = soup.id;

            $apps_with_browsers =
            select
                apps.app as app,
                apps.os as os,
                bro.browsers is not null and ListHas(bro.browsers, 'pp') as has_pp,
                bro.browsers is not null and ListHas(bro.browsers, 'yabro') as has_yabro,
                bro.browsers is not null and ListHas(bro.browsers, 'other') as has_other,
                bro.browsers is not null and ListHasItems(bro.browsers) as has_any,
            from $yabro_pp_apps_indevice_id as apps
            left join any $browsers_by_indevice_id as bro
            using (indevice_id)
            where apps.indevice_id is not null;

            $aps_with_direct_browsers =
            select
                app,
                os,
                direct_browsers is not null and ListHas(direct_browsers, 'pp') as has_pp,
                direct_browsers is not null and ListHas(direct_browsers, 'yabro') as has_yabro,
                direct_browsers is not null and ListHas(direct_browsers, 'other') as has_other,
                direct_browsers is not null and ListHasItems(direct_browsers) as has_any,
            from $yabro_pp_apps_indevice_id;

            -- stats

            $with_direct_browsers =
            select
                app, os,
                count(*) as total_count,
                count_if(has_pp) as has_direct_pp,
                count_if(has_yabro) as has_direct_yabro,
                count_if(has_other) as has_direct_other,
                count_if(has_any) as has_direct_any,
            from $aps_with_direct_browsers
            group by app, os;

            $no_indevice_id =
            select
                app, os,
                count(*) as no_indevice_id
            from $yabro_pp_apps_indevice_id
            where indevice_id is null
            group by app, os;

            $with_indevice_id_and_browsers =
            select
                app, os,
                count(*) as has_indevice_id,
                count_if(has_pp) as has_pp,
                count_if(has_yabro) as has_yabro,
                count_if(has_other) as has_other,
                count_if(has_any) as has_any,
            from $apps_with_browsers
            group by app, os;

            -- finally
            select
                '{date}' as fielddate,
                '{kind}' as kind,
                w.app || '_' || w.os as app,
                no_indevice_id,
                has_indevice_id,
                total_count,
                cast(no_indevice_id as double) / total_count as indevice_id_lost_percent,
                cast(has_indevice_id as double) / total_count as indevice_id_found_percent,
                has_pp,
                cast(has_pp as double) / total_count as has_pp_percent,
                has_yabro,
                cast(has_yabro as double) / total_count as has_yabro_percent,
                has_other,
                cast(has_other as double) / total_count as has_other_percent,
                has_any,
                cast(has_any as double) / total_count as has_any_percent,
                has_direct_pp,
                cast(has_direct_pp as double) / total_count as has_direct_pp_percent,
                has_direct_yabro,
                cast(has_direct_yabro as double) / total_count as has_direct_yabro_percent,
                has_direct_other,
                cast(has_direct_other as double) / total_count as has_direct_other_percent,
                has_direct_any,
                cast(has_direct_any as double) / total_count as has_direct_any_percent,
            from $no_indevice_id as w
            full join $with_indevice_id_and_browsers as wo
            on w.app = wo.app and w.os = wo.os
            full join $with_direct_browsers as direct
            on w.app = direct.app and w.os = direct.os;
        '''.format(
            indevice_last=last,
            idstorage_uuid=self.idstorage(ID_TYPE.UUID),
            idstorage_yandexuid=self.idstorage(ID_TYPE.YANDEXUID),
            soup_yandexuid_uuid=self.soup_matching_dir(ID_TYPE.YANDEXUID, ID_TYPE.UUID),
            date=dt_last,
            kind=self.kind
        ))

    report_config = dedent('''
        ---
        dimensions:
            - fielddate: date
            - kind: string
            - app: string
        measures:
            - no_indevice_id: number
            - has_indevice_id: number
            - total_count: number
            - indevice_id_lost_percent: number
            - indevice_id_found_percent: number
            - has_pp: number
            - has_pp_percent: number
            - has_yabro: number
            - has_yabro_percent: number
            - has_other: number
            - has_other_percent: number
            - has_any: number
            - has_any_percent: number
            - has_direct_pp: number
            - has_direct_pp_percent: number
            - has_direct_yabro: number
            - has_direct_yabro_percent: number
            - has_direct_other: number
            - has_direct_other_percent: number
            - has_direct_any: number
            - has_direct_any_percent: number
    ''')

    solomon_sensors = SolomonSensorConfig(
        sensor_keys=[
            'no_indevice_id',
            'has_indevice_id',
            'total_count',
            'indevice_id_lost_percent',
            'has_pp',
            'has_pp_percent',
            'has_yabro',
            'has_yabro_percent',
            'has_other',
            'has_other_percent',
            'has_any',
            'has_any_percent',
            'has_direct_pp',
            'has_direct_pp_percent',
            'has_direct_yabro',
            'has_direct_yabro_percent',
            'has_direct_other',
            'has_direct_other_percent',
            'has_direct_any',
            'has_direct_any_percent'
        ],
        labels=['kind', 'app']
    )
