# -*- coding: utf-8 -*-
import datetime
import itertools

from textwrap import dedent

from crypta.graph.soup.config.python import (
    EDGE_TYPE as edges,
    LOG_SOURCE as log_source,
)  # N811  # noqa
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import consts
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.juggler.juggler_check_generator import (
    JugglerCheckGenerator,
)
from crypta.lib.python.spine.sandbox import configs, sandbox_scheduler
from crypta.lib.python.spine.solomon import (
    solomon_alert_registry,
    solomon_alert_utils,
    solomon_check_generator,
)
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_latency_metric import YtLatencyMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from sandbox.projects.crypta import merge_to_bigb_collector, run_binary
from sandbox.projects.crypta.graph import (
    bochka,
    export,
    fingerprint,
    households,
    indevice,
    stream,
    styx,
    v1,
)
from sandbox.projects.crypta.graph.metrics.stats import base
from sandbox.projects.crypta.graph.matching import backup
import sandbox.projects.crypta.graph.metrics.telegram.lib as telegram
import sandbox.projects.crypta.graph.metrics.user_params as user_params
from sandbox.projects.crypta.graph.soup import edge_weights
import sandbox.projects.crypta.graph.staff.lib as staff
import sandbox.projects.crypta.graph.rtsklejka as rtsklejka

from crypta.graph.spine import constants
from crypta.graph.spine.constants import (
    TELEGRAM_TAG,
    DATA_IMPORT,
    IDFY,
    CRYPTA_METRICS_TAG,
    MSKOROKHOD,
    ARTEMBELOV,
    ZHEGLOV,
    K_ZAITSEV,
    CRYPTAID,
)
from crypta.graph.spine import bigrt_checks


def notify_break_failure(*logins):
    return sandbox_scheduler.create_email_notifications(
        logins,
        [
            sandbox_scheduler.NotificationStatus.BREAK,
            sandbox_scheduler.NotificationStatus.FAILURE,
        ],
    )


def notify_break_finish(*logins):
    return sandbox_scheduler.create_email_notifications(
        logins,
        [
            sandbox_scheduler.NotificationStatus.BREAK,
            sandbox_scheduler.NotificationStatus.FINISH,
        ],
    )


def notify_failure_exception_timeout(*logins):
    return sandbox_scheduler.create_email_notifications(
        logins,
        [
            sandbox_scheduler.NotificationStatus.FAILURE,
            sandbox_scheduler.NotificationStatus.EXCEPTION,
            sandbox_scheduler.NotificationStatus.TIMEOUT,
        ],
    )


def add_run_binary(sandbox, bundle, args="--date @create_date", **kwargs):
    sandbox.create_scheduler(
        run_binary.CryptaRunBinary,
        extra_params={
            "binary_resource_bundle": bundle,
            "VAULT_OWNER": "CRYPTA",
            "VAULT_YQL_TOKEN": "sec-01csvzgez8yspn3wmr973hhsqk[secret]",
            "VAULT_YT_TOKEN": "sec-01csvzgg3mdasmdygkr5s8n6mz[token]",
            "VAULT_STATFACE_TOKEN": "sec-01daxg3xp40d5pr2s52sb22pkm[secret]",
            "VAULT_SOLOMON_TOKEN": "sec-01e63xgmh8g9fj3c542hrw9vqc[token]",
            "ARGS": args,
            "EXTRA_ENV": {
                "YT_POOL": "crypta_graph",
                "YT_PROXY": "hahn.yt.yandex.net",
            },
        },
        **kwargs
    )


def get_registry():
    juggler = JugglerCheckGenerator(
        host="crypta-graph",
        tags=["crypta-graph", "crypta-idfy-telegram-alert"],
    )

    add_archivator(juggler)
    add_bochka(juggler)
    add_export(juggler)
    add_fingerprint(juggler)
    add_graph_data_import_streams(juggler)
    add_graph_rtsklejka(juggler)
    add_graph_v1(juggler)
    add_households(juggler)
    add_icookie_dict(juggler)
    add_latency_alert(juggler)
    add_merge_to_bigb_collector(juggler)
    add_profile_cleaner(juggler)
    add_soup(juggler)
    add_soupostat_alerts_by_table(juggler)
    add_stats(juggler)
    add_user_params_metrics(juggler)
    add_yt_disk_checks(juggler)
    add_yt_nodes_checks(juggler)
    add_yt_sizes_checks(juggler)
    bigrt_checks.add_bigrt_queue_lag_checks(juggler)
    bigrt_checks.add_bigrt_queue_write_rate_checks(juggler)
    bigrt_checks.add_bigrt_state_checks(juggler)

    return juggler


def add_yt_sizes_checks(juggler):
    yt_registry = yt_config_registry.YtConfigRegistry(
        juggler, yt_config_registry.DEFAULT_ROOTS, "crypta_graph"
    )
    YtSizeMetric(yt_registry, "state/graph", is_recursive=False).add_chunk_count_alert(
        predicate=alert_pb2.GT,
        threshold=int(1.5e6),
    )
    YtSizeMetric(
        yt_registry, "state/indevice", is_recursive=False
    ).add_chunk_count_alert(
        predicate=alert_pb2.GT,
        threshold=int(0.7e6),
    )


def add_merge_to_bigb_collector(juggler):
    juggler_service_suffix = "graph-profiles"
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["GRAPH"])

    for env, pool, subdir in [
        (environment.TESTING, "crypta_graph_testing", environment.TESTING),
        (environment.STABLE, "crypta_graph", environment.PRODUCTION),
    ]:
        scheduler = sandbox.create_scheduler(
            merge_to_bigb_collector.CryptaMergeToBigbCollectorTask,
            kill_timeout=datetime.timedelta(hours=1),
            schedule_interval=datetime.timedelta(minutes=15),
            extra_params={
                "pool": pool,
                "fresh_dir": "//home/crypta/{}/graph/bochka/profiles_log/fresh".format(
                    subdir
                ),
                "output_dir": "//home/crypta/{}/graph/bochka/profiles_log/output".format(
                    subdir
                ),
                "juggler_service_suffix": juggler_service_suffix,
            },
            env=env,
        )

        if env == environment.STABLE:
            scheduler.check(
                crit_time=datetime.timedelta(hours=2),
                juggler_service=merge_to_bigb_collector.CryptaMergeToBigbCollectorTask.get_juggler_service_with_suffix(
                    juggler_service_suffix
                ),
            ).add_yt_dependencies(YtProxy.Group.offline)

    for env, pool, subdir in [
        (environment.STABLE, "crypta_graph", environment.PRODUCTION),
    ]:
        sandbox.create_scheduler(
            merge_to_bigb_collector.CryptaMergeToBigbCollectorTask,
            description="Merge remove profiles log",
            tags=[constants.PROFILE_CLEANER_TAG],
            kill_timeout=datetime.timedelta(hours=1),
            schedule_interval=datetime.timedelta(minutes=15),
            extra_params={
                "pool": pool,
                "fresh_dir": "//home/crypta/{}/graph/bochka/remove_profiles_log/fresh".format(
                    subdir
                ),
                "output_dir": "//home/crypta/{}/graph/bochka/remove_profiles_log/output".format(
                    subdir
                ),
                "juggler_service_suffix": "profile_cleaner",
            },
            env=env,
            notifications=notify_failure_exception_timeout(
                K_ZAITSEV,
            ),
        )


def add_export(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, [IDFY, "GRAPH"])

    for version, recipients in (
        ("v2", [ARTEMBELOV, MSKOROKHOD, ZHEGLOV]),
        ("exp", [ARTEMBELOV]),
    ):
        pool = "crypta_graph_human_matching_{}".format(version)

        sandbox.create_scheduler(
            export.CryptaGraphExport,
            schedule_daily_start_time="2018-04-02T18:00:00Z",
            kill_timeout=datetime.timedelta(days=1),
            extra_params={"pool": pool, "version": version},
            retry_interval=datetime.timedelta(hours=2),
            notifications=notify_break_failure(*recipients),
        ).check(
            crit_time=datetime.timedelta(days=2),
            juggler_service=export.CryptaGraphExport.get_juggler_service_for_version(
                version
            ),
        ).add_yt_dependencies(
            YtProxy.Group.offline
        )


def add_graph_data_import_streams(juggler):
    sandbox = sandbox_scheduler.create_default_generator(
        juggler.clone(tags=["crypta-graph-data-import"]),
        [DATA_IMPORT, "STREAM", "RTCRYPTA"],
    )

    for log_sources, env in (
        (["access"], environment.STABLE),
        (["eal", "redir", "bar"], environment.STABLE),
        (["bs-rtb-log"], environment.STABLE),
        (["wl", "fp"], environment.STABLE),
        (["mm"], environment.STABLE),
        (["postback-log"], environment.STABLE),
        (["access"], environment.TESTING),
        (["eal", "redir", "bar"], environment.TESTING),
        (["bs-rtb-log"], environment.TESTING),
        (["wl", "fp"], environment.TESTING),
        (["mm"], environment.TESTING),
        (["postback-log"], environment.TESTING),
    ):
        add_soup_data_import_scheduler(sandbox, log_sources, env)

    import_tasks = {
        task_name: task_path for task_name, task_path in stream.kSTREAM_IMPORT_CHOICES
    }
    for env, tasks in stream.kSTREAM_IMPORT.items():
        for task_name in tasks:
            add_log_data_import_scheduler(
                sandbox, task_name, import_tasks[task_name], env
            )


def add_soup_data_import_scheduler(sandbox, log_sources, env):
    task = "crypta.graph.data_import.soup.lib.task.SoupTask"

    extra_params = {
        "task": task,
        # "options": {"--yql-embedded-is-embedded": ""},
        "extra_args": {"commit_full_day": "yes"},
    }
    if log_sources != ["all"]:
        extra_params["extra_args"].update({"log_sources": ",".join(log_sources)})

    scheduler = sandbox.create_scheduler(
        stream.CryptaGraphDataImportStream,
        env=env,
        schedule_interval=datetime.timedelta(minutes=30),
        kill_timeout=datetime.timedelta(hours=7),
        extra_params=extra_params,
        retry_interval=datetime.timedelta(minutes=15),
        tags=["SOUP"] + log_sources,
        notifications=notify_break_failure(MSKOROKHOD),
    )
    if env == environment.STABLE:
        scheduler.check(
            crit_time=datetime.timedelta(hours=3),
            juggler_service=stream.CryptaGraphDataImportStream.get_juggler_service_for_task(
                task, log_sources
            ),
        ).add_yt_dependencies(YtProxy.Group.offline)


def add_log_data_import_scheduler(sandbox, task_name, task_path, env):
    retry_intervals = {
        "ExportAccessLog": datetime.timedelta(minutes=60),
        "RTBLog": datetime.timedelta(minutes=30),
    }
    retry_interval = retry_intervals.get(task_name, datetime.timedelta(minutes=10))

    allowed_lags = {
        "ExportAccessLog": "10800",
        "RTBLog": "9000",
    }
    extra_params = {
        "task": task_path,
        "extra_args": {"allowed_lag": allowed_lags.get(task_name, "1800")},
    }

    notifications = None
    if env == environment.STABLE:
        notifications = sandbox_scheduler.create_email_notifications(
            [CRYPTAID],
            [
                sandbox_scheduler.NotificationStatus.EXCEPTION,
                sandbox_scheduler.NotificationStatus.EXPIRED,
                sandbox_scheduler.NotificationStatus.FAILURE,
                sandbox_scheduler.NotificationStatus.TIMEOUT,
            ],
        )

    task_config = configs.TaskConfig(
        stream.CryptaGraphDataImportStream,
        stable=configs.RunConfig(
            schedule_interval=datetime.timedelta(minutes=45),
            kill_timeout=datetime.timedelta(hours=7),
            crit_time=datetime.timedelta(hours=3),
            extra_params=extra_params,
            juggler_service=stream.CryptaGraphDataImportStream.get_juggler_service_for_task(
                task_path, None
            ),
            retry_interval=retry_interval,
            tags=["task/{}".format(task_name)],
            notifications=notifications,
        ),
        testing_from_stable=True,
    )

    if env == environment.STABLE:
        sandbox.create_scheduler(**task_config.stable.get_scheduler_config()).check(
            **task_config.stable.get_check_config()
        ).add_yt_dependencies(YtProxy.Group.offline)
    elif env == environment.TESTING:
        sandbox.create_scheduler(**task_config.testing.get_scheduler_config())
    else:
        raise RuntimeError("Can't create scheduler for env {!r}".format(env))


def add_graph_rtsklejka(juggler):
    sandbox = sandbox_scheduler.create_default_generator(
        juggler, ["RTSKLEJKA", "RTCRYPTA"]
    )
    sandbox.create_scheduler(
        rtsklejka.CryptaGraphRtsklejka,
        description="Convert active crypta_id to mapping",
        extra_params={
            "task": "crypta.graph.rtsklejka.adhoc.lib.tasks.CreateCryptaIdTableTask",
        },
        kill_timeout=datetime.timedelta(hours=3),
        notifications=notify_break_finish(MSKOROKHOD),
        retry_interval=datetime.timedelta(hours=3),
        schedule_daily_start_time="2021-01-26T05:00:00Z",
        sequential_run=False,
    )


def add_yt_disk_checks(juggler_check_generator):
    project_id = "crypta_graph"

    template = dedent(
        """
        let Used = avg({{
            project="yt", cluster="{cluster}", service="accounts",
            account="{account}", sensor="disk_space_in_gb", medium="default"
        }});
        let Limit = avg({{
            project="yt", cluster="{cluster}", service="accounts",
            account="{account}", sensor="disk_space_limit_in_gb", medium="default"
        }});
        let QuotaUsage = (Used/Limit);
        let QuotaUsagePretty = to_fixed(QuotaUsage * 100.0, 2);

        let UsedQuota = to_fixed(Used / 1024.0, 2);
        let LimitQuota = to_fixed(Limit / 1024.0, 2);

        let ThresholdSoft = 0.93;
        let ThresholdHard = 0.95;
        let Account = "{account}";
        let Host = "{cluster}";

        alarm_if(QuotaUsage >= ThresholdHard);
        warn_if(QuotaUsage >= ThresholdSoft);
        """
    )
    juggler_description = dedent(
        """
        {{^is_ok}}
        Please kindly check {{expression.Account}} on {{expression.Host}}:
            quota used {{expression.QuotaUsagePretty}}%
            {{expression.UsedQuota}} Tb out of {{expression.LimitQuota}} Tb
        {{/is_ok}}
        """
    )

    juggler = solomon_check_generator.SolomonCheckGenerator(
        juggler_check_generator.clone(
            host="crypta_yt_accounts", child_group_type=consts.GroupType.host
        ),
        solomon_alert_utils.AlertCreator(
            project_id=project_id,
            selectors={"project": project_id, "service": "accounts"},
        ),
    )

    for cluster, account in (
        ("arnold", "crypta-graph"),
        ("arnold", "crypta-public"),
        ("hahn", "crypta-graph"),
        ("hahn", "crypta-graph-history"),
        ("hume", "crypta-public"),
        ("markov", "crypta-graph"),
        ("zeno", "crypta"),
    ):
        juggler.create_expression_alert_check(
            service="{}.{}".format(cluster, account),
            period=datetime.timedelta(minutes=5),
            program=template.format(cluster=cluster, account=account),
            description=juggler_description,
        ).add_flap_detector(
            FlapDetectorParams(
                datetime.timedelta(minutes=5), datetime.timedelta(minutes=10)
            )
        ).add_tag(
            TELEGRAM_TAG
        )


def add_yt_nodes_checks(juggler_check_generator):
    project_id = "crypta_graph"

    template = dedent(
        """
        let Used = avg({{
            project="yt", cluster="{cluster}", service="accounts",
            account="{account}", sensor="node_count"
        }});
        let Limit = avg({{
            project="yt", cluster="{cluster}", service="accounts",
            account="{account}", sensor="node_count_limit"
        }});
        let QuotaUsage = (Used/Limit);
        let QuotaUsagePretty = to_fixed(QuotaUsage * 100.0, 2);

        let UsedQuota = to_fixed(Used, 2);
        let LimitQuota = to_fixed(Limit, 2);

        let ThresholdSoft = 0.93;
        let ThresholdHard = 0.95;
        let Account = "{account}";
        let Host = "{cluster}";

        alarm_if(QuotaUsage >= ThresholdHard);
        warn_if(QuotaUsage >= ThresholdSoft);
        """
    )
    juggler_description = dedent(
        """
        {{^is_ok}}
        Please kindly check {{expression.Account}} on {{expression.Host}}:
            quota used {{expression.QuotaUsagePretty}}%
            {{expression.UsedQuota}} nodes out of {{expression.LimitQuota}}
        {{/is_ok}}
        """
    )

    juggler = solomon_check_generator.SolomonCheckGenerator(
        juggler_check_generator.clone(
            host="crypta_yt_nodes", child_group_type=consts.GroupType.host
        ),
        solomon_alert_utils.AlertCreator(
            project_id=project_id,
            selectors={"project": project_id, "service": "accounts"},
        ),
    )

    for cluster, account in (
        ("arnold", "crypta-graph"),
        ("arnold", "crypta-public"),
        ("hahn", "crypta-graph"),
        ("hahn", "crypta-graph-history"),
        ("hume", "crypta-public"),
        ("markov", "crypta-graph"),
        ("zeno", "crypta"),
    ):
        juggler.create_expression_alert_check(
            service="{}.{}".format(cluster, account),
            period=datetime.timedelta(minutes=5),
            program=template.format(cluster=cluster, account=account),
            description=juggler_description,
        ).add_flap_detector(
            FlapDetectorParams(
                datetime.timedelta(minutes=5), datetime.timedelta(minutes=10)
            )
        ).add_tag(
            TELEGRAM_TAG
        )


def add_soupostat_alerts_by_table(registry):
    project_id = "crypta_graph"
    creator = solomon_alert_utils.AlertCreator(project_id=project_id)
    solomon = solomon_alert_registry.SolomonAlertRegistry()
    registry.add_subregistry(solomon)

    template = dedent(
        """
        let log_source = '{ls.Name}';
        let series = {{
            cluster='soupostat',
            service='soupostat',
            sensor='{sensor}',
            kind='table'
        }};

        let df = derivative(series);
        let last_val = last(series);
        let max_val = max(series);
        let min_val = min(series);

        // check is monotonously increasing function df > 0
        let min_df = min(df);
        let last_df = last(df);

        // make alerts
        let type = 'no data';
        no_data_if(max_val <= 1);

        let type = 'decrease';
        alarm_if(min_df < 0);

        let type = 'last day equal';
        alarm_if(
            (max_val != min_val)
            && (last_df == 0)
        );

        let type = 'equal';
        warn_if(min_df == 0);
        """
    )

    for ls in log_source.values():
        sensor = "_".join(["*", ls.Name])

        solomon.add_solomon_alert(
            creator.create_expression_alert(
                name="_".join(["sls", ls.Name]),
                period=datetime.timedelta(days=7),
                program=template.format(sensor=sensor, ls=ls),
                group_by_labels=("sensor",),
                juggler_description="",
            )
        )


def add_soupostat_alerts_by_daily(registry):
    project_id = "crypta_graph"
    creator = solomon_alert_utils.AlertCreator(project_id=project_id)
    solomon = solomon_alert_registry.SolomonAlertRegistry()
    registry.add_subregistry(solomon)

    template = dedent(
        """
        let series = shift(
            {{
                cluster='soupostat',
                service='soupostat',
                sensor='{sensor}',
                kind='daily'
            }}, 5d
        );

        let sensor = '{sensor}';
        let current = avg(series);
        let previous = avg(shift(series, 30d));
        let old = avg(shift(series, 90d));

        let percentage = (previous - current) / previous;
        let old_percentage = (old - current) / old;

        let percentage_pretty = to_fixed(percentage * 100, 2);
        let old_percentage_pretty = to_fixed(old_percentage * 100, 2);

        let type = 'strong fall';
        alarm_if(percentage > 0.05);
        alarm_if(old_percentage > 0.03);

        let type = 'light fall';
        warn_if(percentage > 0.03);
        warn_if(old_percentage > 0.01);

        let type = 'ok';
        """
    )

    for _, edge in edges:
        sensor = edges.name(edge)

        solomon.add_solomon_alert(
            creator.create_expression_alert(
                name="daily_{}".format(sensor),
                period=datetime.timedelta(days=7),
                program=template.format(sensor=sensor),
                group_by_labels=("sensor",),
                juggler_description="",
            )
        )


def add_households(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["HOUSEHOLDS"])
    sandbox.create_scheduler(
        households.CryptaHhDataImportWatchlog,
        schedule_interval=datetime.timedelta(hours=4),
        retry_interval=datetime.timedelta(minutes=5),
        kill_timeout=datetime.timedelta(hours=3),
        tags=[DATA_IMPORT],
    )
    sandbox.create_scheduler(
        households.CryptaHhHhMatchPrepare,
        schedule_daily_start_time="2019-01-24T22:00:00Z",
        retry_interval=datetime.timedelta(minutes=25),
        kill_timeout=datetime.timedelta(hours=16),
        notifications=notify_break_failure(MSKOROKHOD),
    )
    sandbox.create_scheduler(
        households.CryptaHhDataImportIncDay,
        schedule_daily_start_time="2018-10-31T20:30:00Z",
        retry_interval=datetime.timedelta(minutes=5),
        kill_timeout=datetime.timedelta(hours=3),
        tags=[DATA_IMPORT],
        notifications=notify_failure_exception_timeout(MSKOROKHOD),
    )


def add_bochka(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, [IDFY, "YT2LB"])

    for tags, config, description, extra_args in (
        (["SOUP"], "/conf/soup.pb.txt", "upload soup to lb", {}),
        (["IDSTORAGE"], "/conf/idstorage.pb.txt", "upload idstorage to lb", {}),
    ):
        sandbox.create_scheduler(
            bochka.CryptaGraphBochka,
            description=description,
            schedule_interval=datetime.timedelta(minutes=15),
            kill_timeout=datetime.timedelta(hours=5),
            tags=tags,
            extra_params=dict(extra_args, config=config, pool="crypta_graph"),
        )


def add_profile_cleaner(juggler):
    sandbox = sandbox_scheduler.create_default_generator(
        juggler, [IDFY, "YT2PROFILE_CLEANER"]
    )

    start_1am = "2022-01-01T01:00:00+03:00"

    for env in [environment.STABLE]:
        sandbox.create_scheduler(
            styx.CryptaGraphStyxExpand,
            description="Run styx expand",
            tags=[constants.STYX_TAG],
            schedule_daily_start_time=start_1am,
            kill_timeout=datetime.timedelta(hours=3),
            env=env,
            extra_params={"pool": constants.CRYPTA_GRAPH},
            notifications=notify_failure_exception_timeout(
                K_ZAITSEV,
            ),
        )


def add_fingerprint(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, [IDFY, "GRAPH"])
    for task in (
        "crypta.graph.fingerprint.match.appmetrica.lib.UpdateSoupEdges",
        "crypta.graph.fingerprint.match.ssp.lib.UpdateSoupEdges",
    ):
        sandbox.create_scheduler(
            fingerprint.CryptaFpBaseTask,
            schedule_interval=datetime.timedelta(days=1),
            kill_timeout=datetime.timedelta(hours=8),
            retry_interval=datetime.timedelta(hours=2),
            extra_params={"task": task},
            notifications=notify_break_failure(MSKOROKHOD),
            sequential_run=False,
        )
    add_run_binary(
        sandbox,
        "CryptaBinaryFingerprintBundle",
        description="Run fingerprint metrics",
        tags=[IDFY, "CRYPTA-13163", CRYPTA_METRICS_TAG],
        schedule_daily_start_time="2020-12-18T17:00:00Z",
        retry_interval=datetime.timedelta(hours=3),
        kill_timeout=datetime.timedelta(hours=7),
        sequential_run=False,
        notifications=notify_break_finish(MSKOROKHOD),
        args="--date @generate_date",
    )


def add_stats(juggler):
    sandbox = sandbox_scheduler.create_default_generator(
        juggler, [IDFY, CRYPTA_METRICS_TAG]
    )

    def create_scheduler_for_metrics(
        version, metric, description, retry_interval, source="//fake"
    ):
        sandbox.create_scheduler(
            base.CryptaStatsBaseMetrics,
            description=description,
            kill_timeout=datetime.timedelta(hours=5),
            schedule_daily_start_time="2019-08-20T06:30:00Z",
            retry_interval=retry_interval,
            sequential_run=False,
            extra_params={
                "date": "yesterday",
                "local": False,
                "source": source,
                "version": version,
                "options": {"--metrics": metric},
            },
            tags=[version.upper()],
        )

    create_scheduler_for_metrics(
        "v2",
        "exp_stats",
        "Calc ExpStats metrics v2 (Money RSYA)",
        datetime.timedelta(minutes=30),
    )
    create_scheduler_for_metrics(
        "v2", "ads", "Calc yql metrics v2 (Money RSYA)", datetime.timedelta(hours=3)
    )
    create_scheduler_for_metrics(
        "v2exp",
        "ads",
        "Calc yql metrics v2exp (Money RSYA)",
        datetime.timedelta(hours=3),
    )

    metrics_for_michurin = " ".join(
        [
            "antispam",
            "devices",
            "entropy",
            "family",
            "id_types",
            "plus",
            "radius",
            "staff",
            "taxi",
            "yuids",
        ]
    )
    create_scheduler_for_metrics(
        "michurin",
        metrics_for_michurin,
        "Calc metrics on michurin_state",
        datetime.timedelta(minutes=30),
        "//home/crypta/production/rtsklejka/state/michurin_state",
    )

    sandbox.create_scheduler(
        telegram.CryptaStatsTelegramNotify,
        schedule_daily_start_time="2020-11-16T08:30:00Z",
        kill_timeout=datetime.timedelta(hours=3),
        retry_interval=datetime.timedelta(minutes=15),
        sequential_run=False,
    )
    sandbox.create_scheduler(
        staff.CryptaStaffExporter,
        schedule_daily_start_time="2018-02-07T08:30:00Z",
        kill_timeout=datetime.timedelta(hours=3),
        sequential_run=False,
        extra_params={
            "pool": "crypta_graph",
        },
    )
    add_run_binary(
        sandbox,
        "CryptaBinaryLoginMetricsBundle",
        description="Run login metrics CRYPTR-193",
        tags=["CRYPTR-193"],
        kill_timeout=datetime.timedelta(hours=5),
        schedule_daily_start_time="2018-11-22T01:44:00Z",
        retry_interval=datetime.timedelta(hours=4),
        sequential_run=False,
        notifications=notify_break_finish(IDFY),
        args="--date @generate_date",
    )


def add_icookie_dict(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, [IDFY])
    add_run_binary(
        sandbox,
        "CryptaBinaryIcookieDictBundle",
        description="Make yuid-icookie dicts",
        tags=["CRYPTR-620", "CRYPTA-SOUP2DICT"],
        kill_timeout=datetime.timedelta(hours=3),
        schedule_daily_start_time="2018-09-20T00:00:00Z",
        sequential_run=False,
        args="",
    )


def add_soup(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, [IDFY, "GRAPH"])

    sandbox.create_scheduler(
        edge_weights.CryptaGraphSoupEdgeWeightsHistograms,
        description="Prepare survival model",
        schedule_interval=datetime.timedelta(days=4),
        kill_timeout=datetime.timedelta(hours=8),
        extra_params={
            "pool": "crypta_graph",
            "source_dir": "//home/crypta/production/state/graph/v2/soup",
            "destination_path": "//home/crypta/production/state/graph/v2/soup/cooked/stats/HistogramEdges",
        },
    )
    sandbox.create_scheduler(
        indevice.CryptaSoupyIndeviceTimedCheckFast,
        description="Check that fast indevice table exists by 14:00",
        kill_timeout=datetime.timedelta(hours=1),
        schedule_daily_start_time="2021-02-25T14:00:00+03:00",
        retry_interval=datetime.timedelta(hours=1),
        sequential_run=False,
        extra_params={"hour": 14},
    )


def add_graph_v1(juggler):
    sandbox = sandbox_scheduler.create_default_generator(
        juggler, [IDFY, "CRYPTA-MATCHING"]
    )

    # NOTE (v1 waits for v2 anyway, so we're starting it late)
    start_1am = "2021-01-01T01:00:00+03:00"
    start_time = {
        "v2": start_1am,
    }

    for run_type, env in itertools.product(
        ("v2",),
        (
            environment.TESTING,
            environment.STABLE,
        ),
    ):
        sandbox.create_scheduler(
            v1.CryptaGraphV1,
            description="Run skleyka v1 in SB",
            tags=[run_type.upper()],
            start_immediately=True,
            kill_timeout=datetime.timedelta(days=4),
            schedule_daily_start_time=start_time[run_type],
            env=env,
            notifications=notify_failure_exception_timeout(
                K_ZAITSEV,
                MSKOROKHOD,
            ),
            sequential_run=False,
            extra_params={
                "date": "yesterday",
                "run_type": run_type,
            },
        ).check(
            crit_time=datetime.timedelta(days=2),
            juggler_service=v1.CryptaGraphV1.get_juggler_service_for_run_type(
                run_type, env
            ),
        )


def add_user_params_metrics(juggler):
    sandbox = sandbox_scheduler.create_default_generator(
        juggler, [IDFY, CRYPTA_METRICS_TAG]
    )

    sandbox.create_scheduler(
        user_params.CryptaUserParamsMetrics,
        description="Metrics for user params",
        kill_timeout=datetime.timedelta(hours=5),
        notifications=notify_break_finish(IDFY),
        schedule_daily_start_time="2021-08-27T08:00:00Z",
        sequential_run=False,
        extra_params={
            "output_dir": "//home/crypta/production/graph/metrics/user_params"
        },
    )


def add_latency_alert(juggler):

    root_paths = {
        environment.PRODUCTION: "//home/crypta/production/state/graph/v2",
        environment.TESTING: "//home/crypta/production/state/graph/v2exp",
    }

    path_and_timeout_list = [
        ("matching/vertices_no_multi_profile_by_id_type", datetime.timedelta(days=2)),
        ("matching/edges_by_crypta_id", datetime.timedelta(days=2)),
        ("indevice/full/last", datetime.timedelta(days=3)),
        ("indevice/fast/last", datetime.timedelta(hours=32)),
    ]

    yt_registry = yt_config_registry.YtConfigRegistry(
        juggler,
        root_paths,
        yt_config_registry.DEFAULT_SOLOMON_PROJECT,
    )

    for path, timeout in path_and_timeout_list:
        YtLatencyMetric(yt_registry, path).add_age_alert(
            threshold=timeout,
        )

    return juggler


def add_archivator(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, [IDFY])

    def create_scheduler_for_archivator(start_at, conf, descr):
        sandbox.create_scheduler(
            backup.CryptaGraphHumanMatchingBackup,
            description=descr,
            kill_timeout=datetime.timedelta(hours=5),
            schedule_daily_start_time=start_at,
            retry_interval=datetime.timedelta(minutes=30),
            sequential_run=False,
            extra_params={
                "config": conf,
            },
            tags=[],
        )

    create_scheduler_for_archivator(
        "2022-07-19T01:30:00Z", "/conf/fpc.pb.txt", "Backup fpc IndexDuid"
    )
    create_scheduler_for_archivator(
        "2022-07-19T02:30:00Z", "/conf/he.pb.txt", "Dump Herschel state"
    )
    create_scheduler_for_archivator(
        "2022-07-19T03:30:00Z", "/conf/mi.pb.txt", "Dump Michurin state"
    )
    create_scheduler_for_archivator(
        "2022-07-19T03:30:00Z", "/conf/mi_cid.pb.txt", "Dump Michurin Cid state"
    )
