import datetime
from textwrap import dedent

from crypta.lib.python.spine.juggler import consts
from crypta.lib.python.spine.solomon import solomon_alert_utils, solomon_check_generator
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.graph.spine.constants import TELEGRAM_TAG, CRYPTA_GRAPH


def add_bigrt_state_checks(juggler_check_generator):
    template = dedent(
        """
        let TotalRaw = {{
            project="{project_id}",
            cluster="{cluster}",
            service="{cluster}",
            host="cluster",
            sensor="epochs_total",
            system="{system}"
            }};
        let FailedRaw = {{
            project="{project_id}",
            cluster="{cluster}",
            service="{cluster}",
            host="cluster",
            sensor="epochs_failed",
            system="{system}"
            }};

        let TotalAvg = avg(group_lines('avg', TotalRaw));
        let FailRate = FailedRaw / (TotalAvg + 1e-2);

        let SumFailRate = group_lines('sum', FailRate);

        let SumP80 = percentile(80, SumFailRate);

        let Value = to_fixed(SumP80, 3);
        let Cluster = "{cluster}";
        let System = "{system}";

        alarm_if(SumP80 > 0.1);
        warn_if(SumP80 > 0.05);
        """
    )
    juggler_description = dedent(
        """
        {{^is_ok}}
        Bigrt states '{{expression.System}}' are failing for {{expression.Cluster}}.
        Fail rate is {{expression.Value}}.
        {{/is_ok}}
        """
    )

    juggler = solomon_check_generator.SolomonCheckGenerator(
        juggler_check_generator.clone(host="rtsklejka.bigrt", child_group_type=consts.GroupType.host),
        solomon_alert_utils.AlertCreator(
            project_id=CRYPTA_GRAPH, selectors={"project": CRYPTA_GRAPH, "service": "bigrt_state_health"}
        ),
    )

    for cluster, system in (
        ("michurin-prod", "michurin"),
        ("michurin-prod", "cryptaid"),
        ("michurin-prod", "vulture"),
        ("michurin-prod", "vultureExp"),

        ("michurin-test", "michurin"),
        ("michurin-test", "cryptaid"),
        ("michurin-test", "vulture"),
        ("michurin-test", "vultureExp"),
    ):
        juggler.create_expression_alert_check(
            service="state.fail.{}.{}".format(cluster, system),
            period=datetime.timedelta(minutes=5),
            program=template.format(cluster=cluster, project_id=CRYPTA_GRAPH, system=system),
            description=juggler_description,
        ).add_flap_detector(FlapDetectorParams(datetime.timedelta(minutes=5), datetime.timedelta(minutes=10))).add_tag(
            TELEGRAM_TAG
        )


def add_bigrt_queue_lag_checks(juggler_check_generator):
    template = dedent(
        """
        let QueueLag = avg({{
            project="big_rt_queue_daemon",
            service="public_queue_daemon",
            cluster="{cluster}",
            queue="{queue}",
            consumer="{consumer}",
            host="cluster",
            sensor="row_lag",
            shard="-"
        }});

        let Queue = "{queue}";
        let Cluster = "{cluster}";
        let Mln = 1000 * 1000;
        let ThresholdSoft = 10 * Mln;
        let ThresholdHard = 40 * Mln;
        let QueueLagMln = to_fixed(QueueLag/Mln, 2);

        alarm_if(QueueLag >= ThresholdHard);
        warn_if(QueueLag >= ThresholdSoft);
        """
    )
    juggler_description = dedent(
        """
        {{^is_ok}}
        Queue lag for {{expression.Queue}} on {{expression.Cluster}} is {{expression.QueueLagMln}} mln rows
        {{/is_ok}}
        """
    )

    juggler = solomon_check_generator.SolomonCheckGenerator(
        juggler_check_generator.clone(host="rtsklejka.bigrt", child_group_type=consts.GroupType.host),
        solomon_alert_utils.AlertCreator(
            project_id=CRYPTA_GRAPH, selectors={"project": CRYPTA_GRAPH, "service": "public_queue_daemon"}
        ),
    )

    for cluster, consumer, queue in (
        ("markov", "michurin", "//home/crypta/production/rtsklejka/qyt/sharded_events"),
        ("markov", "michurin_rewind", "//home/crypta/production/rtsklejka/qyt/rewound_events"),
        ("markov", "michurin_cryptaid", "//home/crypta/production/rtsklejka/qyt/cryptaid_events"),
        ("markov", "michurin_vulture", "//home/crypta/production/rtsklejka/qyt/vulture_events"),
        ("markov", "michurin_vulture_exp", "//home/crypta/production/rtsklejka/qyt/vulture_exp_events"),

        ("hahn", "michurin", "//home/crypta/testing/rtsklejka/qyt/sharded_events"),
        ("hahn", "michurin_rewind", "//home/crypta/testing/rtsklejka/qyt/rewound_events"),
        ("hahn", "michurin_cryptaid", "//home/crypta/testing/rtsklejka/qyt/cryptaid_events"),
        ("hahn", "michurin_vulture", "//home/crypta/testing/rtsklejka/qyt/vulture_events"),
        ("hahn", "michurin_vulture_exp", "//home/crypta/testing/rtsklejka/qyt/vulture_exp_events"),
    ):
        juggler.create_expression_alert_check(
            service="queue.lag.{}.{}".format(cluster, consumer),
            period=datetime.timedelta(minutes=5),
            program=template.format(cluster=cluster, consumer=consumer, queue=queue),
            description=juggler_description,
        ).add_flap_detector(FlapDetectorParams(datetime.timedelta(minutes=5), datetime.timedelta(minutes=10))).add_tag(
            TELEGRAM_TAG
        )


def add_bigrt_queue_write_rate_checks(juggler_check_generator):
    template = dedent(
        """
        let QueueWriteRate = avg({{
            project="big_rt_queue_daemon",
            service="public_queue_daemon",
            cluster="{cluster}",
            queue="{queue}",
            consumer="ideal_consumer",
            host="cluster",
            sensor="row_rate",
            shard="-"
        }});

        let Queue = "{queue}";
        let Cluster = "{cluster}";
        let Threshold = 100;
        let QueueWriteRateK = to_fixed(QueueWriteRate/1000, 2);

        warn_if(QueueWriteRate < Threshold);
        """
    )
    juggler_description = dedent(
        """
        {{^is_ok}}
        Queue write rate is low for {{expression.Queue}} on {{expression.Cluster}}: {{expression.QueueWriteRateK}}K rows
        {{/is_ok}}
        """
    )

    juggler = solomon_check_generator.SolomonCheckGenerator(
        juggler_check_generator.clone(host="rtsklejka.bigrt", child_group_type=consts.GroupType.host),
        solomon_alert_utils.AlertCreator(
            project_id=CRYPTA_GRAPH, selectors={"project": CRYPTA_GRAPH, "service": "public_queue_daemon"}
        ),
    )

    for cluster, queue in (
        ("markov", "//home/crypta/production/rtsklejka/qyt/sharded_events"),
        ("markov", "//home/crypta/production/rtsklejka/qyt/rewound_events"),
        ("markov", "//home/crypta/production/rtsklejka/qyt/cryptaid_events"),
        ("markov", "//home/crypta/production/rtsklejka/qyt/vulture_events"),
        ("markov", "//home/crypta/production/rtsklejka/qyt/vulture_exp_events"),

        ("hahn", "//home/crypta/testing/rtsklejka/qyt/sharded_events"),
        ("hahn", "//home/crypta/testing/rtsklejka/qyt/rewound_events"),
        ("hahn", "//home/crypta/testing/rtsklejka/qyt/cryptaid_events"),
        ("hahn", "//home/crypta/testing/rtsklejka/qyt/vulture_events"),
        ("hahn", "//home/crypta/testing/rtsklejka/qyt/vulture_exp_events"),
    ):
        short_queue_name = queue.rsplit('/', 1)[-1]
        juggler.create_expression_alert_check(
            service="queue.write_rate.{}.{}".format(cluster, short_queue_name),
            period=datetime.timedelta(minutes=5),
            program=template.format(cluster=cluster, queue=queue),
            description=juggler_description,
        ).add_flap_detector(FlapDetectorParams(datetime.timedelta(minutes=5), datetime.timedelta(minutes=10))).add_tag(
            TELEGRAM_TAG
        )
