import contextlib
import datetime
import logging

from cached_property import cached_property

import crypta.lib.python.bt.conf.conf as conf  # noqa
from crypta.lib.python.bt.tasks import TransactionalYtTask
from crypta.lib.python.bt.workflow import IndependentTask, Parameter
from crypta.lib.python.yql_runner.base_parser import BaseParser
from crypta.lib.python.yql_runner.task import ConfMixin

import yt.wrapper as yt

MATCHING_PATH = '//home/crypta/{crypta_env}/state/graph/v2/matching'
STYX_SOURCE_PATH = '//home/crypta/{crypta_env}/styx/export/raw-events'
BASE_PATH = '//home/crypta/{crypta_env}/styx/graph'

logger = logging.getLogger(__name__)


class Runner(BaseParser):
    QUERY_TEMPLATE = 'query.sql.j2'


class StyxExpandTask(IndependentTask, TransactionalYtTask, ConfMixin):
    """Expand puids from styx into known ids."""

    last_table_name = Parameter(default='last_table')
    base_path = Parameter(default=BASE_PATH)

    update_last_table = Parameter(
        default="True",
        parse=lambda value: value.lower() in {"+", "yes", "true", "t", "1"}
    )

    @cached_property
    def styx_target_path(self):
        return yt.ypath_join(self.styx_export_path, 'export')

    @cached_property
    def last_table_path(self):
        return yt.ypath_join(self.styx_export_path, self.last_table_name)

    @cached_property
    def styx_export_path(self):
        return self.base_path.format(crypta_env=self.crypta_env)

    @cached_property
    def styx_source_path(self):
        return STYX_SOURCE_PATH.format(crypta_env=self.crypta_env)

    @cached_property
    def vertices_by_type_path(self):
        return yt.ypath_join(MATCHING_PATH, 'vertices_no_multi_profile_by_id_type').format(crypta_env=self.crypta_env)

    @cached_property
    def vertices_by_crypta_id_path(self):
        return yt.ypath_join(MATCHING_PATH, 'vertices_no_multi_profile_by_crypta_id').format(crypta_env=self.crypta_env)

    def run(self, *args, **kwargs):
        pool = conf.proto.Yt.Pool
        proxy = conf.proto.Yt.Proxy

        table_name = datetime.datetime.now().strftime("%Y-%m-%dT%H:%M:%S")
        target_full_table_path = yt.ypath_join(self.styx_target_path, 'full', table_name)
        target_bb_table_path = yt.ypath_join(self.styx_target_path, 'styx2bb', table_name)

        task = Runner(
            date=None,
            yt_proxy=proxy,
            pool=pool,
            is_embedded=False,
        )
        task.run(
            styx_source_dir=self.styx_source_path,
            target_full_table_path=target_full_table_path,
            target_bb_table_path=target_bb_table_path,
            last_table_path=self.last_table_path,

            by_id_type_table_path=self.vertices_by_type_path,
            by_crypta_id_table_path=self.vertices_by_crypta_id_path,

            update_last_table=self.update_last_table,
        )

    @contextlib.contextmanager
    def run_context(self):
        with super(StyxExpandTask, self).run_context() as ctx:
            yield ctx
