import luigi

from textwrap import dedent

from crypta.graph.soup.config.python import (
    EDGE_TYPE as edges,
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
)

from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.lib.luigi import base_luigi_task, yt_luigi
from crypta.graph.v1.python.utils.yql_utils import run_yql
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyDumpTable

YQL = dedent(
    """
    $orders = '{orders}';
    $output = '{output}';

    $email_type = '{email_type}';
    $phone_type = '{phone_type}';
    $puid_type = '{puid_type}';
    $uuid_type = '{uuid_type}';
    $yuid_type = '{yuid_type}';

    $log_source = '{log_source}';
    $source_type = '{source_type}';

    $pair_types = AsList(
        {edge_types}
    );

    --==============================================================================

    $make_pairs_closure = ($dict) -> {{
        RETURN ($pair) -> {{
            $id1Type = $pair.0;
            $id2Type = $pair.1;
            $id1 = $dict[$id1Type];
            $id2 = $dict[$id2Type];
            return if (
                ($id1 is not null and $id2 is not null),
                AsStruct(
                    $id1Type as id1Type,
                    $id1 as id1,
                    $id2Type as id2Type,
                    $id2 as id2
                ),
                null
            );
        }};
    }};

    $get_dict = ($id1Type, $id1, $id2Type, $id2) -> {{
        $dict = ToDict(AsList(($id1Type, $id1), ($id2Type, $id2)));
        return $dict;
    }};

    --==============================================================================

    $data = (
        select
            fact.userIds.email as email,
            fact.userIds.phone as phone,
            cast(fact.userIds.puid as string) as puid,
            fact.userIds.`uuid` as `uuid`,
            fact.userIds.yandexuid as yuid,
            `timestamp` as ts
        from $orders
    );

    --==============================================================================

    $email_yuid = (
        select $get_dict($email_type, unwrap(email), $yuid_type, unwrap(yuid)) as `dict`, ts
        from $data
        where (email is not null) and (yuid is not null)
    );

    $email_uuid = (
        select $get_dict($email_type, unwrap(email), $uuid_type, unwrap(`uuid`)) as `dict`, ts
        from $data
        where (email is not null) and (`uuid` is not null)
    );

    $email_puid = (
        select $get_dict($email_type, unwrap(email), $puid_type, unwrap(puid)) as `dict`, ts
        from $data
        where (email is not null) and (puid is not null)
    );

    $email_phone = (
        select $get_dict($email_type, unwrap(email), $phone_type, unwrap(phone)) as `dict`, ts
        from $data
        where (email is not null) and (phone is not null)
    );

    $yuid_puid = (
        select $get_dict($yuid_type, unwrap(yuid), $puid_type, unwrap(puid)) as `dict`, ts
        from $data
        where (yuid is not null) and (puid is not null)
    );

    $yuid_phone = (
        select $get_dict($yuid_type, unwrap(yuid), $phone_type, unwrap(phone)) as `dict`, ts
        from $data
        where (yuid is not null) and (phone is not null)
    );

    $uuid_puid = (
        select $get_dict($uuid_type, unwrap(`uuid`), $puid_type, unwrap(puid)) as `dict`, ts
        from $data
        where (`uuid` is not null) and (puid is not null)
    );

    $uuid_phone = (
        select $get_dict($uuid_type, unwrap(`uuid`), $phone_type, unwrap(phone)) as `dict`, ts
        from $data
        where (`uuid` is not null) and (phone is not null)
    );

    $puid_phone = (
        select $get_dict($puid_type, unwrap(puid), $phone_type, unwrap(phone)) as `dict`, ts
        from $data
        where (puid is not null) and (phone is not null)
    );

    --==============================================================================

    $all = (
        select * from $email_yuid
        union all
        select * from $email_uuid
        union all
        select * from $email_puid
        union all
        select * from $email_phone
        union all
        select * from $yuid_puid
        union all
        select * from $yuid_phone
        union all
        select * from $uuid_puid
        union all
        select * from $uuid_phone
        union all
        select * from $puid_phone
    );

    $all_pairs = (
        select ListFlatMap($pair_types, $make_pairs_closure(`dict`)) as pair_list, ts
        from $all
    );

    --==============================================================================

    INSERT INTO $output WITH TRUNCATE
    SELECT
        pair_list.id1 as id1,
        pair_list.id1Type as id1Type,
        pair_list.id2 as id2,
        pair_list.id2Type as id2Type,
        $source_type as sourceType,
        $log_source as logSource,
        CAST(DateTime::MakeDate(DateTime::FromMilliseconds(ts)) AS String) AS `date`
    FROM $all_pairs
    FLATTEN LIST BY pair_list;
"""
)


def mk_yql_edge_types(edge_types):
    return ",\n".join(("('{}', '{}')".format(x.Id1Type.Name, x.Id2Type.Name) for x in edge_types))


class ImportMarketOrders(base_luigi_task.BaseTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportMarketOrders, self).__init__(*args, **kwargs)
        self.log_source = log_source.MARKET
        self.log_table = log_source.MARKET.LogPath
        self.soup_daily_dump_tables = SoupDailyDumpTable(self.log_source, self.date)
        self.market_order_edges = [
            edge_type
            for edge_type in edges.values()
            if edge_type.LogSource == self.log_source
            and edge_type.Props.ActivityType == edge_type.Props.DAILY
            and edge_type.Props.SupplyMethod == edge_type.Props.DUMP
        ]
        if not self.market_order_edges:
            raise Exception("not founded market order edges")

    def requires(self):
        return yt_luigi.ExternalInput(self.log_table, allow_empty=True)

    def output(self):
        return self.soup_daily_dump_tables.as_target()

    def run(self):
        self.soup_daily_dump_tables.ensure_dir()
        soup_tmp_table = self.soup_daily_dump_tables.create()
        query = YQL.format(
            orders=self.log_table,
            output=soup_tmp_table,
            email_type=ids.EMAIL.Name,
            phone_type=ids.PHONE.Name,
            puid_type=ids.PUID.Name,
            uuid_type=ids.UUID.Name,
            yuid_type=ids.YANDEXUID.Name,
            log_source=self.log_source.Name,
            source_type=source_type.ORDERS.Name,
            edge_types=mk_yql_edge_types(self.market_order_edges),
        )
        run_yql(query=query)
        mr.set_generate_date(soup_tmp_table, self.date)
