from functools import partial

import luigi

from yt.wrapper import ypath

from crypta.graph.v1.python.data_imports.day_aggregate import reduce_yuid_log_events_day
from crypta.lib.python.identifiers.identifiers import GenericID
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.utils import yt_clients
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDumpTable

from crypta.graph.soup.config.python import (
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


MULTI_PHONE_PAYMENT_LIMIT = 1


def map_yamoney_v1(rec, table_indexes):
    yuid = rec.get("PAYER_ENTITY_YANDEX_UID")
    puid = rec.get("PAYER_ENTITY_UID")

    def defined(str_value):
        if str_value and str_value != "0":
            return True
        else:
            return False

    peiw = rec["PAYER_ENTITY_IS_WALLET"]
    if peiw == "True":
        is_login = True
    else:
        is_login = False

    if defined(yuid) or defined(puid):

        explicit_phone = rec.get("phone")
        explicit_email = rec.get("email")

        if explicit_phone:
            if defined(puid):  # prefer puid first
                yield {
                    "puid": puid,
                    "id_value": explicit_phone,
                    "id_type": config.ID_TYPE_PHONE,
                    "is_login": is_login,
                    "explicit": True,
                    "@table_index": table_indexes[(config.ID_TYPE_PUID, config.ID_TYPE_PHONE)],
                }
            elif defined(yuid):
                yield {
                    "yuid": yuid,
                    "id_value": explicit_phone,
                    "id_type": config.ID_TYPE_PHONE,
                    "is_login": is_login,
                    "explicit": True,
                    "@table_index": table_indexes[(config.ID_TYPE_YUID, config.ID_TYPE_PHONE)],
                }
        if explicit_email:
            if defined(puid):  # prefer puid first
                yield {
                    "puid": puid,
                    "id_value": explicit_email,
                    "id_type": config.ID_TYPE_EMAIL,
                    "is_login": is_login,
                    "explicit": True,
                    "@table_index": table_indexes[(config.ID_TYPE_PUID, config.ID_TYPE_EMAIL)],
                }
            elif defined(yuid):
                yield {
                    "yuid": yuid,
                    "id_value": explicit_email,
                    "id_type": config.ID_TYPE_EMAIL,
                    "is_login": is_login,
                    "explicit": True,
                    "@table_index": table_indexes[(config.ID_TYPE_YUID, config.ID_TYPE_EMAIL)],
                }

        id_value = rec.get("PAYER_ENTITY_IDENTIFIER")
        ident_type = rec.get("PAYER_ENTITY_IDENTIFIER_TYPE")
        if id_value and ident_type and ident_type != "False":
            id_type_mapping = {
                "1": config.ID_TYPE_PHONE,
                "2": config.ID_TYPE_YAMONEY_CARD_TOKEN,
                "3": config.ID_TYPE_EMAIL,
                "4": config.ID_TYPE_YAMONEY_ACCOUNT,
            }
            id_type = id_type_mapping[ident_type]

            if id_type == config.ID_TYPE_YAMONEY_ACCOUNT:
                id_value = utils.md5(id_value)

            if defined(puid):  # prefer puid first
                yield {
                    "puid": puid,
                    "id_value": id_value,
                    "id_type": id_type,
                    "is_login": is_login,
                    "explicit": False,
                    "@table_index": table_indexes[(config.ID_TYPE_PUID, id_type)],
                }
            elif defined(yuid):
                yield {
                    "yuid": yuid,
                    "id_value": id_value,
                    "id_type": id_type,
                    "is_login": is_login,
                    "explicit": False,
                    "@table_index": table_indexes[(config.ID_TYPE_YUID, id_type)],
                }


def map_yamoney_v2(
    rec, phone_yamoneyid_yamoney, email_yamoneyid_yamoney, puid_yamoneyid_yamoney, yuid_yamoneyid_yamoney
):
    yamoney_type = rec["id_type"]

    out_rec = {"yamoney_id": rec["yamoney_id"]}

    if yamoney_type == 1:
        out_rec["id_value"] = rec["id_value"]
        out_rec["id_type"] = config.ID_TYPE_PHONE
        out_rec["@table_index"] = 0
        yield SoupDumpTable.make_rec(
            rec["id_value"], str(rec["yamoney_id"]), phone_yamoneyid_yamoney, [], table_index=7
        )
    elif yamoney_type == 2:
        out_rec["id_value"] = rec["id_value"]
        out_rec["id_type"] = config.ID_TYPE_YAMONEY_CARD_TOKEN
        out_rec["@table_index"] = 1
        # ingnore, do not throw to soup
    elif yamoney_type == 3:
        out_rec["id_value"] = rec["id_value"]
        out_rec["id_type"] = config.ID_TYPE_EMAIL
        out_rec["@table_index"] = 2
        yield SoupDumpTable.make_rec(
            rec["id_value"], str(rec["yamoney_id"]), email_yamoneyid_yamoney, [], table_index=8
        )
    elif yamoney_type == 4:
        out_rec["id_value"] = rec["id_value"]
        out_rec["id_type"] = config.ID_TYPE_YAMONEY_ACCOUNT
        out_rec["@table_index"] = 3
        # ingnore, do not throw to soup
    elif yamoney_type == 7:
        out_rec["id_value"] = rec["id_value"]
        out_rec["id_type"] = "yamoney_internal"
        out_rec["@table_index"] = 4
        # ingnore, do not throw to soup
    elif yamoney_type == 5:
        out_rec["puid"] = rec["id_value"]
        out_rec["@table_index"] = 5
        yield SoupDumpTable.make_rec(
            rec["id_value"], str(rec["yamoney_id"]), puid_yamoneyid_yamoney, [], table_index=9
        )
    elif yamoney_type == 6:
        out_rec["yuid"] = rec["id_value"]
        out_rec["@table_index"] = 6
        yield SoupDumpTable.make_rec(
            rec["id_value"], str(rec["yamoney_id"]), yuid_yamoneyid_yamoney, [], table_index=10
        )
    else:
        raise Exception("unsupported yamoney type %s" % yamoney_type)

    yield out_rec


def reduce_by_yamoney_id(yamoney_id_key, recs, table_indexes):
    recs = list(recs)
    puids = set()
    yuids = set()
    id_recs = []
    for r in recs:
        if "puid" in r:
            puids.add(r["puid"])
        elif "yuid" in r:
            yuids.add(r["yuid"])
        else:
            id_recs.append(r)

    for r in id_recs:
        id_type = r["id_type"]
        for puid in puids:
            r[config.ID_TYPE_PUID] = puid
            r["@table_index"] = table_indexes[(config.ID_TYPE_PUID, id_type)]
            yield r

        for yuid in yuids:
            r["yuid"] = yuid
            r["@table_index"] = table_indexes[(config.ID_TYPE_YUID, id_type)]
            yield r


def map_phone(rec, yuid_phone_yamoney):
    yuid = rec["yuid"].strip()
    phone_raw = "+7" + rec["phone"]
    check_phone = GenericID("phone", phone_raw)
    phone = check_phone.normalize if check_phone.is_valid() else None
    if phone:
        yield {"yuid": yuid, "id_value": phone, "id_type": config.ID_TYPE_PHONE}
        yield SoupDumpTable.make_rec(yuid, phone_raw[1:], yuid_phone_yamoney, [], table_index=2)
    else:
        rec["@table_index"] = 1
        yield rec


def filter_multi_yuid_phones(phone_key, recs):
    recs = list(recs)
    if len(recs) <= MULTI_PHONE_PAYMENT_LIMIT:
        table_index = 0
    else:
        table_index = 1

    for r in recs:
        r["@table_index"] = table_index
        yield r


def import_yamoney_dump_v1(in_table, workdir, matching_types, dt=None):

    out_tables = [workdir + "%s_%s" % (l, r) for l, r in matching_types]
    table_indexes = {t: idx for idx, t in enumerate(matching_types)}

    yt_client = yt_clients.get_yt_client()
    yt_client.run_map(partial(map_yamoney_v1, table_indexes=table_indexes), in_table, out_tables + [workdir + "debug"])


def import_yamoney_dump_v2(in_dump_table, workdir, matching_types, soup_tables):

    tmp_tables = [
        workdir + config.ID_TYPE_PHONE,
        workdir + config.ID_TYPE_YAMONEY_CARD_TOKEN,
        workdir + config.ID_TYPE_EMAIL,
        workdir + config.ID_TYPE_YAMONEY_ACCOUNT,
        workdir + config.ID_TYPE_YAMONEY_INTERNAL,
        workdir + "puid",
        workdir + "yuid",
    ]

    yt_client = yt_clients.get_yt_client()

    with yt_client.Transaction() as tr:
        soup_paths = [t.create(tr) for t in soup_tables]
        yt_client.run_map(
            partial(
                map_yamoney_v2,
                phone_yamoneyid_yamoney=soup_tables[0].edge_type,
                email_yamoneyid_yamoney=soup_tables[1].edge_type,
                puid_yamoneyid_yamoney=soup_tables[2].edge_type,
                yuid_yamoneyid_yamoney=soup_tables[3].edge_type,
            ),
            in_dump_table,
            tmp_tables + soup_paths,
        )

        out_tables = [workdir + "%s_%s" % (l, r) for l, r in matching_types]
        table_indexes = {t: idx for idx, t in enumerate(matching_types)}

        mr.sort_all(tmp_tables, sort_by="yamoney_id")
        yt_client.run_reduce(
            partial(reduce_by_yamoney_id, table_indexes=table_indexes), tmp_tables, out_tables, reduce_by="yamoney_id"
        )

        SoupDumpTable.finalize_all(soup_tables)

    return


def import_yamoney_phone_dump(in_dump_table, workdir, soup_table):

    yt_client = yt_clients.get_yt_client()
    with yt_client.Transaction() as tr:

        yt_client.run_map(
            partial(map_phone, yuid_phone_yamoney=soup_table.edge_type),
            in_dump_table,
            [workdir + "yuid_phone_good", workdir + "yuid_phone_bad", soup_table.create(tr)],
            job_count=30,
        )

        yt_client.run_map_reduce(
            None,
            filter_multi_yuid_phones,
            workdir + "yuid_phone_good",
            [workdir + "yuid_phone", workdir + "yuid_phone_multi"],
            reduce_by="yuid",
        )

        soup_table.finalize()


class ImportYandexMoneyDump(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {"yamoney_dict": config.GRAPH_YT_DICTS_FOLDER + "yamoney/"}

    def output_folders(self):
        return {"yuid_raw": config.GRAPH_YT_DICTS_FOLDER + "yuid_raw/"}

    def __init__(self, *args, **kwargs):
        super(ImportYandexMoneyDump, self).__init__(*args, **kwargs)

        self.matching_types = [
            (config.ID_TYPE_PUID, config.ID_TYPE_PHONE),
            (config.ID_TYPE_PUID, config.ID_TYPE_EMAIL),
            (config.ID_TYPE_PUID, config.ID_TYPE_YAMONEY_ACCOUNT),
            (config.ID_TYPE_PUID, config.ID_TYPE_YAMONEY_CARD_TOKEN),
            (config.ID_TYPE_PUID, config.ID_TYPE_YAMONEY_INTERNAL),
            (config.ID_TYPE_YUID, config.ID_TYPE_PHONE),
            (config.ID_TYPE_YUID, config.ID_TYPE_EMAIL),
            (config.ID_TYPE_YUID, config.ID_TYPE_YAMONEY_ACCOUNT),
            (config.ID_TYPE_YUID, config.ID_TYPE_YAMONEY_CARD_TOKEN),
            (config.ID_TYPE_YUID, config.ID_TYPE_YAMONEY_INTERNAL),
        ]

        self.phone_yamoneyid_yamoney = SoupDumpTable(
            edges.get_edge_type(ids.PHONE, ids.YAMONEY_ID, source_type.YANDEX_MONEY, log_source.YANDEX_MONEY_DUMP),
            self.date,
        )
        self.email_yamoneyid_yamoney = SoupDumpTable(
            edges.get_edge_type(ids.EMAIL, ids.YAMONEY_ID, source_type.YANDEX_MONEY, log_source.YANDEX_MONEY_DUMP),
            self.date,
        )
        self.puid_yamoneyid_yamoney = SoupDumpTable(
            edges.get_edge_type(ids.PUID, ids.YAMONEY_ID, source_type.YANDEX_MONEY, log_source.YANDEX_MONEY_DUMP),
            self.date,
        )
        self.yuid_yamoneyid_yamoney = SoupDumpTable(
            edges.get_edge_type(ids.YANDEXUID, ids.YAMONEY_ID, source_type.YANDEX_MONEY, log_source.YANDEX_MONEY_DUMP),
            self.date,
        )

        self.soup_tables = [
            self.phone_yamoneyid_yamoney,
            self.email_yamoneyid_yamoney,
            self.puid_yamoneyid_yamoney,
            self.yuid_yamoneyid_yamoney,
        ]

        self.yuid_phone_soup_v2 = SoupDumpTable(
            edges.get_edge_type(ids.YANDEXUID, ids.PHONE, source_type.YANDEX_MONEY, log_source.YANDEX_MONEY_DUMP),
            self.date,
        )

    def requires(self):
        return [
            yt_luigi.ExternalInput(self.in_f("yamoney_dict") + "yamoney_in_v1"),
            yt_luigi.ExternalInput(self.in_f("yamoney_dict") + "yamoney_in_v2"),
            yt_luigi.ExternalInput(self.in_f("yamoney_dict") + "yamoney_phone_payment"),
        ]

    def before_run(self):
        mr.mkdir(ypath.ypath_join(self.out_f("yuid_raw"), "yamoney", "v1"))
        mr.mkdir(ypath.ypath_join(self.out_f("yuid_raw"), "yamoney", "v2"))
        mr.mkdir(ypath.ypath_join(self.out_f("yuid_raw"), "yamoney", "phone_payment"))

    def run(self):
        yuid_raw = self.out_f("yuid_raw")
        workdir = yuid_raw + "yamoney/"

        import_yamoney_dump_v1(self.in_f("yamoney_dict") + "yamoney_in_v1", workdir + "v1/", self.matching_types)

        import_yamoney_dump_v2(
            self.in_f("yamoney_dict") + "yamoney_in_v2",
            workdir + "v2/",
            self.matching_types,
            soup_tables=self.soup_tables,
        )

        import_yamoney_phone_dump(
            self.in_f("yamoney_dict") + "yamoney_phone_payment", workdir + "phone_payment/", self.yuid_phone_soup_v2
        )

        ops = []

        # Also there is no activity dates info in dict, thus we don't specify date
        for left_id, right_id in self.matching_types:
            v1_table = workdir + "v1/%s_%s" % (left_id, right_id)
            v2_table = workdir + "v2/%s_%s" % (left_id, right_id)
            v3_table = workdir + "phone_payment/%s_%s" % (left_id, right_id)

            op = self.yt.run_map_reduce(
                None,
                partial(
                    reduce_yuid_log_events_day,
                    dt=None,
                    key_col=left_id,
                    id_type=right_id,
                    source_type=config.ID_SOURCE_TYPE_YAMONEY,
                ),
                [t for t in [v1_table, v2_table, v3_table] if self.yt.exists(t)],
                yuid_raw + left_id + "_with_" + right_id + "_" + config.ID_SOURCE_TYPE_YAMONEY,
                sort_by=left_id,
                reduce_by=left_id,
                sync=False,
            )
            ops.append(op)

        utils.wait_all(ops)

        for left_id, right_id in self.matching_types:
            mr.set_generate_date(
                yuid_raw + left_id + "_with_" + right_id + "_" + config.ID_SOURCE_TYPE_YAMONEY, self.date
            )

    def output(self):
        soup_targets = [t.as_target() for t in self.soup_tables + [self.yuid_phone_soup_v2]]

        return [
            yt_luigi.YtDateTarget(
                self.out_f("yuid_raw") + left_id + "_with_" + right_id + "_" + config.ID_SOURCE_TYPE_YAMONEY, self.date
            )
            for left_id, right_id in self.matching_types
        ] + soup_targets


def read_phone_dump():
    with open("1702.txt") as f:
        for line in f.readlines():
            values = line.split("\t")
            phone = values[0]
            yuid = values[1].strip()
            yield {"phone": phone, "yuid": yuid}
