from functools import partial

import luigi

from crypta.graph.v1.python.data_imports.day_aggregate import (
    reduce_device_log_events_day,
    finalize_device_yuid_day_tables,
)
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.utils import yt_clients
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable

from crypta.graph.soup.config.python import (
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


def map_oauth(rec, puid_uuid_oauth, puid_mm_device_id_oauth):
    uuid = rec.get("uuid", "")
    device_id = utils.norm_id(rec.get("device_id", ""))
    uid = rec.get("uid", "")
    soup_table_index = 3

    try:
        ts = int(rec.get("unixtime", ""))
    except ValueError:
        rec["@table_index"] = 2
        yield rec
        return

    if uuid and uuid != "0" and uid:
        yield {
            config.ID_TYPE_UUID: uuid,
            config.ID_TYPE_PUID: uid,
            "ts": abs(ts),
            "source": config.ID_SOURCE_TYPE_PASSPORT_OAUTH,
        }  # 'uuid' 'puid'
        yield SoupDailyLogTable.make_rec(uid, uuid, puid_uuid_oauth, ts=abs(ts), table_index=soup_table_index)

    if device_id and uid:
        yield {
            "mmetric_devid": device_id,
            config.ID_TYPE_PUID: uid,
            "ts": abs(ts),
            "source": config.ID_SOURCE_TYPE_PASSPORT_OAUTH,
            "@table_index": 1,
        }  # 'devid' 'puid'
        yield SoupDailyLogTable.make_rec(
            uid, device_id, puid_mm_device_id_oauth, ts=abs(ts), table_index=soup_table_index
        )


def import_oauth(oauth_log, dt, workdir, devid_raw_f, soup_log):

    yt_client = yt_clients.get_yt_client()
    yt_client.run_map(
        partial(
            map_oauth,
            puid_uuid_oauth=edges.get_edge_type(
                ids.PUID, ids.UUID, source_type.APP_PASSPORT_AUTH, log_source.OAUTH_LOG
            ),
            puid_mm_device_id_oauth=edges.get_edge_type(
                ids.PUID, ids.MM_DEVICE_ID, source_type.APP_PASSPORT_AUTH, log_source.OAUTH_LOG
            ),
        ),
        oauth_log,
        [
            workdir + "uuid_puid_oauth_tmp",
            workdir + "mmetric_devid_puid_oauth_tmp",
            workdir + "oauth_bad",
            soup_log.create(),
        ],
    )

    soup_log.prepare_daily_tables_from_log()

    utils.wait_all(
        [
            yt_client.run_sort(
                workdir + "uuid_puid_oauth_tmp", sort_by=[config.ID_TYPE_PUID, config.ID_TYPE_UUID, "ts"], sync=False
            ),
            yt_client.run_sort(
                workdir + "mmetric_devid_puid_oauth_tmp",
                sort_by=[config.ID_TYPE_PUID, config.ID_TYPE_MMETRIC_DEVID, "ts"],
                sync=False,
            ),
        ]
    )

    uuid_puid_oauth = devid_raw_f + "uuid_puid_" + config.ID_SOURCE_TYPE_PASSPORT_OAUTH
    mmetric_devid_puid_oauth = devid_raw_f + "mmetric_devid_puid_" + config.ID_SOURCE_TYPE_PASSPORT_OAUTH

    utils.wait_all(
        [
            yt_client.run_reduce(
                partial(reduce_device_log_events_day, dt=dt, source_type=config.ID_SOURCE_TYPE_PASSPORT_OAUTH),
                workdir + "uuid_puid_oauth_tmp",
                uuid_puid_oauth,
                sort_by=[config.ID_TYPE_PUID, config.ID_TYPE_UUID, "ts"],
                reduce_by=[config.ID_TYPE_PUID, config.ID_TYPE_UUID],
                sync=False,
            ),
            yt_client.run_reduce(
                partial(reduce_device_log_events_day, dt=dt, source_type=config.ID_SOURCE_TYPE_PASSPORT_OAUTH),
                workdir + "mmetric_devid_puid_oauth_tmp",
                mmetric_devid_puid_oauth,
                sort_by=[config.ID_TYPE_PUID, config.ID_TYPE_MMETRIC_DEVID, "ts"],
                reduce_by=[config.ID_TYPE_PUID, config.ID_TYPE_MMETRIC_DEVID],
                sync=False,
            ),
        ]
    )

    finalize_device_yuid_day_tables([uuid_puid_oauth, mmetric_devid_puid_oauth])

    mr.drop(workdir + "uuid_puid_oauth_tmp")
    mr.drop(workdir + "mmetric_devid_puid_oauth_tmp")


class ImportOauthLogsDayTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportOauthLogsDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.OAUTH_LOG, self.date)

    def input_folders(self):
        return {"logfeller": config.LOG_FOLDERS["oauth"]}

    def output_folders(self):
        return {
            "raw": config.INDEVICE_YT_FOLDER + self.date + "/perfect/devid_raw_day/",
            "logs": config.INDEVICE_YT_FOLDER + self.date + "/perfect/logs/",
        }

    def requires(self):
        return [yt_luigi.ExternalInput(self.in_f("logfeller") + self.date)]

    def before_run(self):
        mr.mkdir(self.out_f("raw"))
        mr.mkdir(self.out_f("logs"))
        self.soup_log.ensure_dir()

    def run(self):
        oauth_log = config.LOG_FOLDERS["oauth"] + self.date

        import_oauth(oauth_log, self.date, self.out_f("logs"), self.out_f("raw"), self.soup_log)

    def output(self):

        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        out_date_folder = self.out_f("raw")
        return [
            yt_luigi.YtTarget(t)
            for t in [
                out_date_folder + "uuid_puid_" + config.ID_SOURCE_TYPE_PASSPORT_OAUTH,
                out_date_folder + "mmetric_devid_puid_" + config.ID_SOURCE_TYPE_PASSPORT_OAUTH,
            ]
        ] + soup_out_tables
