import datetime
from textwrap import dedent

import luigi

from crypta.graph.v1.python.data_imports import StreamImportBaseTask, soup_table

from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils.yql_utils import run_yql

from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
)


class ImportRedirLogsDayTask(StreamImportBaseTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()
    observed_logs = ["common-redir-log"]

    def __init__(self, *args, **kwargs):
        ls = log_source.REDIR_LOG
        wait_keys = ["RedirLogImportTask", "SoupTask"]

        super(ImportRedirLogsDayTask, self).__init__(ls, wait_keys, *args, **kwargs)

    def output_folders(self):
        return {
            "devid_raw": config.INDEVICE_YT_FOLDER + self.date + "/perfect/devid_raw_day/",
            "logs": config.INDEVICE_YT_FOLDER + self.date + "/perfect/logs/",
        }

    def run_stream(self):
        day_ts = int(
            (datetime.datetime.strptime(self.date, "%Y-%m-%d") - datetime.datetime(1970, 1, 1)).total_seconds()
        )

        template = dedent(
            """SELECT NULL; -- for python inline sql code highlight

                PRAGMA yt.ExternalTx = '{tx}';

                $uuid_yuid_soup = (
                    SELECT
                        id1 AS yuid,
                        id2 AS `uuid`,
                        'redir' AS source_type,
                        '{dt}' AS dt,
                        1 AS hits_count,
                        {day_ts} AS last_ts
                    FROM CONCAT(
                        `{uuid_yuid_soup_mobile}`,
                        `{uuid_yuid_soup_desktop}`
                    ) WITH COLUMNS Struct<
                        `dates`: List<String>?
                    >
                    WHERE ListHas(dates, '{dt}')
                );

                INSERT INTO `{output}` WITH TRUNCATE
                SELECT * FROM $uuid_yuid_soup
                ORDER BY `uuid`, yuid;
            """
        )
        context = dict(
            dt=self.date,
            uuid_yuid_soup_mobile=soup_table(ids.YANDEXUID, ids.UUID, source_type.APP_URL_REDIR, log_source.REDIR_LOG),
            uuid_yuid_soup_desktop=soup_table(
                ids.YANDEXUID, ids.UUID, source_type.APP_URL_REDIR_DESKTOP, log_source.REDIR_LOG
            ),
            day_ts=day_ts,
            output="".join([self.out_f("devid_raw"), "uuid_yuid_", config.ID_SOURCE_TYPE_REDIR]),
        )

        with self.yt.Transaction() as tx:
            run_yql(query=template.format(tx=tx.transaction_id, **context))

    def output(self):
        out_date_folder = self.out_f("devid_raw")
        output_tables = [out_date_folder + "uuid_yuid_" + config.ID_SOURCE_TYPE_REDIR]
        return super(ImportRedirLogsDayTask, self).output() + [yt_luigi.YtTarget(table) for table in output_tables]
