from functools import partial

import luigi

from crypta.graph.v1.python.data_imports.day_aggregate import (
    reduce_device_log_events_day,
    finalize_device_yuid_day_tables,
)
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.utils import uat_utils
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable

from crypta.graph.soup.config.python import (
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


def version_tuple(v):
    return tuple(map(int, (v.split("."))))


def map_sdk(rec, yuid_uuid_sdk):
    uuid = rec.get("uuid", "")
    host = rec.get("host")
    normal_index = 0
    soup_index = 1

    cookies = rec.get("cookies", "")
    yuid = mr.get_field_value("yandexuid", cookies, ";")
    if not yuid:
        yuid = mr.get_field_value("yandexuid", cookies, "; ")

    timestamp = rec.get("timestamp", "")
    tz = rec.get("timezone", "")
    try:
        ts = utils.get_ts(timestamp, tz)
    except ValueError:
        ts = ""

    user_agent = rec.get("user_agent")

    if uuid and yuid and ts and user_agent:
        if host and host == "redirect.appmetrica.yandex.ru":  # only RU cookies

            ua = uat_utils.Ua(user_agent)
            os = ua.profile_info.get("OSFamily", "").lower()
            os_version = ua.profile_info.get("OSVersion", "").lower()

            if os and os_version:

                if os == "ios" and version_tuple(os_version)[0] > 10:
                    return

                yield {
                    "yuid": yuid,
                    "uuid": uuid,
                    "ts": abs(ts),
                    "source": config.ID_SOURCE_TYPE_SDK,
                    "@table_index": normal_index,
                }
                yield SoupDailyLogTable.make_rec(yuid, uuid, yuid_uuid_sdk, ts=ts, table_index=soup_index)


class ImportSdkLogsDayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportSdkLogsDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.MOBILE_REDIRECT_BIND_ID_LOG, self.date)

    def input_folders(self):
        return {"logfeller": config.LOGFELLER_SDK_FOLDER}

    def output_folders(self):
        return {"raw": config.INDEVICE_YT_FOLDER + self.date + "/perfect/devid_raw_day/"}

    def requires(self):
        return [yt_luigi.ExternalInput(self.in_f("logfeller") + self.date)]

    def before_run(self):
        tmp_logs_f = config.INDEVICE_YT_FOLDER + self.date + "/perfect/logs/"
        mr.mkdir(tmp_logs_f)
        mr.mkdir(self.out_f("raw"))
        self.soup_log.ensure_dir()

    def run(self):
        tmp_logs_f = config.INDEVICE_YT_FOLDER + self.date + "/perfect/logs/"
        out_f = self.out_f("raw")

        self.yt.run_map(
            partial(
                map_sdk,
                yuid_uuid_sdk=edges.get_edge_type(
                    ids.YANDEXUID, ids.UUID, source_type.APP_METRICA_SDK, log_source.MOBILE_REDIRECT_BIND_ID_LOG
                ),
            ),
            self.in_f("logfeller") + self.date,
            [tmp_logs_f + "tmp_soup_yuid_uuid_bind_id_log", self.soup_log.create()],
        )
        uuid_yuid_sdk = out_f + "uuid_yuid_" + config.ID_SOURCE_TYPE_SDK

        self.yt.run_map_reduce(
            None,
            partial(reduce_device_log_events_day, dt=self.date, source_type=config.ID_SOURCE_TYPE_SDK),
            config.INDEVICE_YT_FOLDER + self.date + "/perfect/logs/tmp_soup_yuid_uuid_bind_id_log",
            uuid_yuid_sdk,
            sort_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID, "ts"],
            reduce_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID],
        )

        finalize_device_yuid_day_tables([uuid_yuid_sdk])

        self.soup_log.prepare_daily_tables_from_log()

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        return [
            yt_luigi.YtTarget(t) for t in [self.out_f("raw") + "uuid_yuid_" + config.ID_SOURCE_TYPE_SDK]
        ] + soup_out_tables
