from functools import partial

import luigi

from crypta.graph.v1.python.data_imports.day_aggregate import (
    reduce_device_log_events_day,
    finalize_device_yuid_day_tables,
)
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import yt_clients
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable

from crypta.graph.soup.config.python import (
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


def to_list(s):
    s = s.replace("'", "")
    s = s.replace(" ", "")
    s = s.replace("[", "")
    s = s.replace("]", "")

    return s.split(",")


def map_tracking(rec, yuid_uuid_mobtrack):
    event_type = rec.get("EventType", "")
    if event_type != "EVENT_AD_INSTALL":
        return

    attribution_method = rec.get("AttributionMethod")
    if attribution_method == "fingerprint":
        # CRYPTR-164
        return

    uuid = rec.get("UUID")
    click_timestamp = rec.get("ClickTimestamp")

    if click_timestamp and uuid:
        url_keys = to_list(rec.get("UrlParameters_Keys", ""))

        if "yandexuid" in url_keys:  # conflicts with YandexUidRu - 2 different yuids. 46k rows at 29.04.17
            i = url_keys.index("yandexuid")
            url_values = to_list(rec.get("UrlParameters_Values", ""))
            if i < len(url_values):
                yuid = url_values[i]
                if yuid:
                    ts = int(click_timestamp)
                    yield {
                        config.ID_TYPE_UUID: uuid,
                        config.ID_TYPE_YUID: yuid,
                        "ts": abs(ts),
                        "source": config.ID_SOURCE_TYPE_TRACK,
                    }
                    yield SoupDailyLogTable.make_rec(yuid, uuid, yuid_uuid_mobtrack, ts=abs(ts), table_index=1)
        yuid = rec.get("YandexUidRu")
        if yuid:
            ts = int(click_timestamp)
            yield {
                config.ID_TYPE_UUID: uuid,
                config.ID_TYPE_YUID: yuid,
                "ts": abs(ts),
                "source": config.ID_SOURCE_TYPE_TRACK,
            }
            yield SoupDailyLogTable.make_rec(yuid, uuid, yuid_uuid_mobtrack, ts=abs(ts), table_index=1)


def import_tracking_log(tracking_logs, dt, workdir, devid_raw_f, soup_log):

    log_t = workdir + "%s_%s_%s_tmp" % (config.ID_TYPE_UUID, config.ID_TYPE_YUID, config.ID_SOURCE_TYPE_TRACK)
    out_t = devid_raw_f + "%s_%s_%s" % (config.ID_TYPE_UUID, config.ID_TYPE_YUID, config.ID_SOURCE_TYPE_TRACK)

    yt_client = yt_clients.get_yt_client()
    yt_client.run_map(
        partial(
            map_tracking,
            yuid_uuid_mobtrack=edges.get_edge_type(
                ids.YANDEXUID, ids.UUID, source_type.APP_METRICA_INSTALL_TRACKING, log_source.MOBILE_TRACKING_LOG
            ),
        ),
        tracking_logs,
        [log_t, soup_log.create()],
    )

    soup_log.prepare_daily_tables_from_log()

    yt_client.run_sort(log_t, sort_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID, "ts"])

    yt_client.run_reduce(
        partial(reduce_device_log_events_day, dt=dt, source_type=config.ID_SOURCE_TYPE_TRACK),
        log_t,
        out_t,
        sort_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID, "ts"],
        reduce_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID],
    )

    finalize_device_yuid_day_tables([out_t])


class ImportMobileTrackingLogsDayTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportMobileTrackingLogsDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.MOBILE_TRACKING_LOG, self.date)

    def requires(self):
        return [
            yt_luigi.ExternalInput(config.LOG_FOLDERS["mob_tracking"] + self.date),
            yt_luigi.ExternalInput(config.LOG_FOLDERS["mob_tracking_private"] + self.date),
        ]

    def before_run(self):
        out_indevice_folder = config.INDEVICE_YT_FOLDER + self.date + "/"
        out_date_folder = out_indevice_folder + "perfect/"
        mr.mkdir(out_date_folder)
        mr.mkdir(out_date_folder + "logs/")
        mr.mkdir(out_date_folder + "devid_raw_day/")
        self.soup_log.ensure_dir()

    def run(self):
        out_indevice_folder = config.INDEVICE_YT_FOLDER + self.date + "/"
        out_date_folder = out_indevice_folder + "perfect/"

        tracking_logs = [r.table for r in self.requires()]

        import_tracking_log(
            tracking_logs, self.date, out_date_folder + "logs/", out_date_folder + "devid_raw_day/", self.soup_log
        )

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        out_indevice_folder = config.INDEVICE_YT_FOLDER + self.date + "/"
        out_date_folder = out_indevice_folder + "perfect/devid_raw_day/"
        return [
            yt_luigi.YtTarget(
                out_date_folder + "%s_%s_%s" % (config.ID_TYPE_UUID, config.ID_TYPE_YUID, config.ID_SOURCE_TYPE_TRACK)
            )
        ] + soup_out_tables
