import logging

from datetime import datetime
from textwrap import dedent

import luigi

from crypta.graph.v1.python.data_imports import StreamImportBaseTask, soup_table
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils.yql_utils import run_yql

from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
)

logger = logging.getLogger(__name__)


class ImportAccessLogsDayTask(StreamImportBaseTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    resources = {"import_access_log_lock": 1}

    def __init__(self, *args, **kwargs):
        ls = log_source.ACCESS_LOG
        wait_keys = ["AccessLogImportTask", "SoupTask"]

        super(ImportAccessLogsDayTask, self).__init__(ls, wait_keys, *args, **kwargs)

    def input_folders(self):
        return {"logs": config.ACCESS_LOGS_FOLDER}

    def output_folders(self):
        return {
            "devid_raw": config.INDEVICE_YT_FOLDER + self.date + "/perfect/devid_raw_day/",
            "yuid_raw": config.YT_OUTPUT_FOLDER + self.date + "/yuid_raw/",
            "raw_links": config.YT_OUTPUT_FOLDER + self.date + "/raw_links/",
        }

    def before_run(self):
        mr.mkdir(self.out_f("devid_raw"))
        mr.mkdir(self.out_f("yuid_raw"))

    @property
    def observed_logs(self):
        return self.in_f("logs")

    def run_stream(self):
        template = dedent(
            """SELECT NULL; -- for python inline sql code highlight

            PRAGMA yt.ExternalTx = '{tx}';

            $uuid_yuid_soup = (
                select id1 as yuid, id2 as `uuid`
                from `{uuid_yuid_soup}`
                WITH COLUMNS Struct<
                    `dates`: List<String>?
                >
                where ListHas(dates, '{dt}')
            );

            $devid_yuid_soup = (
                select id1 as yuid, id2 as mmetric_devid
                from `{devid_yuid_soup}`
                WITH COLUMNS Struct<
                    `dates`: List<String>?
                >
                where ListHas(dates, '{dt}')
            );

            $ua = (
                select distinct yandexuid, user_agent, os, dt
                from CONCAT({ua_tables})
            );

            insert into `{uuid_yuid_access}` with truncate
            select
                `uuid`,
                yuid,
                -{day_ts} as ts,
                'access_log' as source_type,
                1 AS hits_count,
                '{dt}' AS dt
            from $uuid_yuid_soup;

            insert into `{mmetric_devid_yuid_yp_ios}` with truncate
            select
                a.yuid as yuid,
                a.mmetric_devid as mmetric_devid,
                ABS({day_ts}) as last_ts,
                b.os as os,
                '{dt}' AS dt,
                1 AS hits_count,
                'access_yp_did_ios' AS source_type
            from $devid_yuid_soup as a
            left join $ua as b on a.yuid = b.yandexuid
            where String::ToLower(b.os) = 'ios';

            insert into `{mmetric_devid_yuid_yp_android}` with truncate
            select
                a.yuid as yuid,
                a.mmetric_devid as mmetric_devid,
                ABS({day_ts}) as last_ts,
                b.os as os,
                '{dt}' AS dt,
                1 AS hits_count,
                'access_yp_did_android' AS source_type
            from $devid_yuid_soup as a
            left join $ua as b on a.yuid = b.yandexuid
            where String::ToLower(b.os) != 'ios';

            insert into `{yuid_with_ua_access_log}` with truncate
            select
                yandexuid as yuid,
                user_agent as id_value,
                '{dt}' AS id_date,
                'ua' AS id_type,
                1 AS id_count,
                'access_log' AS source_type
            from $ua
            where dt = '{dt}';

            insert into `{yuid_ua_wrong_date}` with truncate
            select * from $ua where dt != '{dt}';
        """
        )

        with self.yt.Transaction() as tx:
            wrong_date_table_name = "wrong_date"
            access_log_ua_dir = self.conf.paths.stream.extra_data + "/AccessLogImportTask/access_log_ua/"
            ua_tables = self.yt.list(access_log_ua_dir.rstrip("/"), absolute=True)

            context = dict(
                tx=tx.transaction_id,
                uuid_yuid_soup=soup_table(ids.YANDEXUID, ids.UUID, source_type.APP_URL_REDIR, log_source.ACCESS_LOG),
                devid_yuid_soup=soup_table(
                    ids.YANDEXUID, ids.MM_DEVICE_ID, source_type.ACCESS_YP_COOKIE, log_source.ACCESS_LOG
                ),
                dt=self.date,
                ua_tables=", ".join(["`{}`".format(x) for x in ua_tables]),
                day_ts=int((datetime.strptime(self.date, "%Y-%m-%d") - datetime(1970, 1, 1)).total_seconds()),
                uuid_yuid_access=self.out_f("devid_raw") + "uuid_yuid_" + config.ID_SOURCE_TYPE_ACCESS_LOG,
                mmetric_devid_yuid_yp_ios=(
                    self.out_f("devid_raw") + "mmetric_devid_yuid_" + config.ID_SOURCE_TYPE_ACCESS_YP_IOS
                ),
                mmetric_devid_yuid_yp_android=(
                    self.out_f("devid_raw") + "mmetric_devid_yuid_" + config.ID_SOURCE_TYPE_ACCESS_YP_ANDROID
                ),
                yuid_with_ua_access_log=self.out_f("yuid_raw") + "yuid_with_ua_access_log",
                yuid_ua_wrong_date=access_log_ua_dir + wrong_date_table_name,
            )

            run_yql(query=template.format(**context))

            for t in ua_tables:
                if not t.endswith(wrong_date_table_name):
                    self.yt.remove(t)

    def output(self):
        uuid_yuid_table = self.out_f("devid_raw") + "uuid_yuid_" + config.ID_SOURCE_TYPE_ACCESS_LOG
        yuid_ua_table = (
            self.out_f("yuid_raw") + "yuid_with_" + config.FIELD_UA + "_" + config.ID_SOURCE_TYPE_ACCESS_LOG
        )
        return super(ImportAccessLogsDayTask, self).output() + [
            yt_luigi.YtTarget(t) for t in [uuid_yuid_table, yuid_ua_table]
        ]
