import datetime
from textwrap import dedent

import luigi

from crypta.graph.v1.python.data_imports import StreamImportBaseTask, soup_table
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils.yql_utils import run_yql

from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
)


class ImportBarNavigDayTask(StreamImportBaseTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    resources = {"import_bar_navig_lock": 1}
    observed_logs = ["bar-navig-log"]

    def __init__(self, *args, **kwargs):
        ls = log_source.BAR_NAVIG_LOG
        wait_keys = ["BarNavigImportTask", "SoupTask"]
        super(ImportBarNavigDayTask, self).__init__(ls, wait_keys, *args, **kwargs)

    def run_stream(self):
        template = dedent(
            """SELECT NULL; -- for python inline sql code highlight

            PRAGMA yt.ExternalTx = '{tx}';

            INSERT INTO `{out_bar_r1}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'r1' AS id_type,
                id2 AS id_value,
                'barlog' AS source_type
            FROM `{soup_r1_yasoft}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;

            INSERT INTO `{out_bar_ui}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'ui' AS id_type,
                id2 AS id_value,
                'barlog' AS source_type
            FROM `{soup_ui_yasoft}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;

            INSERT INTO `{out_yabro_android}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS dt,
                1 AS hits_count,
                -{day_ts} AS last_ts,
                id2 AS `uuid`,
                'yabrowser_android' AS source_type
            FROM `{soup_uuid_android}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            -- ORDER BY yuid, dt
            ;

            INSERT INTO `{out_vk_bar}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'vk' AS id_type,
                id2 AS id_value,
                'barlog' AS source_type
            FROM `{soup_vk_yabro}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;

            INSERT INTO `{out_ok_bar}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'ok' AS id_type,
                id2 AS id_value,
                'barlog' AS source_type
            FROM `{soup_ok_yabro}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;

            INSERT INTO `{out_em_bar}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'email' AS id_type,
                id2 AS id_value,
                'barlog' AS source_type
            FROM `{soup_em_yabro}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;
        """
        )

        context = dict(
            # staff
            dt=self.date,
            day_ts=int(datetime.datetime.strptime(self.date, "%Y-%m-%d").strftime("%s")),
            # input tables
            soup_r1_yasoft=soup_table(ids.YANDEXUID, ids.DISTR_R1, source_type.YASOFT, log_source.BAR_NAVIG_LOG),
            soup_ui_yasoft=soup_table(ids.YANDEXUID, ids.DISTR_UI, source_type.YASOFT, log_source.BAR_NAVIG_LOG),
            soup_uuid_android=soup_table(ids.YANDEXUID, ids.UUID, source_type.YABRO_ANDROID, log_source.BAR_NAVIG_LOG),
            soup_vk_yabro=soup_table(ids.YANDEXUID, ids.VK_ID, source_type.YABRO_AJAX, log_source.BAR_NAVIG_LOG),
            soup_ok_yabro=soup_table(ids.YANDEXUID, ids.OK_ID, source_type.YABRO_AJAX, log_source.BAR_NAVIG_LOG),
            soup_em_yabro=soup_table(ids.YANDEXUID, ids.EMAIL, source_type.YABRO_AJAX, log_source.BAR_NAVIG_LOG),
            # output tables
            out_bar_r1=self._yt_out_path(config.ID_TYPE_BAR_R1),
            out_bar_ui=self._yt_out_path(config.ID_TYPE_BAR_UI),
            out_yabro_android=self._indev_out_path(config.ID_SOURCE_TYPE_YABROWSER_ANDROID),
            out_vk_bar=self._yt_out_path(config.ID_TYPE_VKCOM + "_" + config.ID_SOURCE_TYPE_BARLOG),
            out_ok_bar=self._yt_out_path(config.ID_TYPE_OKRU + "_" + config.ID_SOURCE_TYPE_BARLOG),
            out_em_bar=self._yt_out_path(config.ID_TYPE_EMAIL + "_" + config.ID_SOURCE_TYPE_BARLOG),
        )

        with self.yt.Transaction() as tx:
            run_yql(query=template.format(tx=tx.transaction_id, **context))

    def _yt_out_path(self, id_type):
        return "".join((config.YT_OUTPUT_FOLDER, self.date, "/yuid_raw/yuid_with_", id_type))

    def _indev_out_path(self, id_type):
        return "".join((config.INDEVICE_YT_FOLDER, self.date, "/perfect/devid_raw_day/uuid_yuid_", id_type))

    def output(self):
        return super(ImportBarNavigDayTask, self).output() + [
            yt_luigi.YtTarget(self._yt_out_path(config.ID_TYPE_BAR_UI)),
            yt_luigi.YtTarget(self._yt_out_path(config.ID_TYPE_BAR_R1)),
            yt_luigi.YtTarget(self._indev_out_path(config.ID_SOURCE_TYPE_YABROWSER_ANDROID)),
            yt_luigi.YtTarget(self._yt_out_path(config.ID_TYPE_VKCOM + "_" + config.ID_SOURCE_TYPE_BARLOG)),
            yt_luigi.YtTarget(self._yt_out_path(config.ID_TYPE_OKRU + "_" + config.ID_SOURCE_TYPE_BARLOG)),
            yt_luigi.YtTarget(self._yt_out_path(config.ID_TYPE_EMAIL + "_" + config.ID_SOURCE_TYPE_BARLOG)),
        ]
