import datetime

from textwrap import dedent

import luigi

from crypta.graph.v1.python.data_imports import StreamImportBaseTask, soup_table
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils.yql_utils import run_yql

from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
)


class ImportEalDayTask(StreamImportBaseTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    observed_logs = ["export-access-log"]

    def __init__(self, *args, **kwargs):
        ls = log_source.EXPORT_ACCESS_LOG
        wait_keys = ["EalImportTask", "SoupTask"]
        super(ImportEalDayTask, self).__init__(ls, wait_keys, *args, **kwargs)

    def output_folders(self):
        return {"raw": config.YT_OUTPUT_FOLDER + self.date + "/yuid_raw/"}

    def before_run(self):
        mr.mkdir(self.out_f("raw"))

    def run_stream(self):
        template = dedent(
            """SELECT NULL; -- for python inline sql code highlight

            PRAGMA yt.ExternalTx = '{tx}';

            INSERT INTO `{out_eal}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'ui' AS id_type,
                id2 AS id_value,
                'eal' AS source_type
            FROM `{soup_yasoft}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;

            INSERT INTO `{out_punto}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'ui' AS id_type,
                id2 AS id_value,
                'punto' AS source_type
            FROM `{soup_yasoft_punto}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;

            INSERT INTO `{out_ext_bro}` WITH TRUNCATE
            SELECT
                id1 AS yuid,
                '{dt}' AS id_date,
                1 AS id_count,
                'ui' AS id_type,
                id2 AS id_value,
                'ext_bro' AS source_type
            FROM `{soup_external_bro}`
            WITH COLUMNS Struct<`dates`: List<String>?>
            WHERE ListHas(dates, '{dt}')
            ORDER BY yuid, id_date;
        """
        )

        context = dict(
            # staff
            dt=self.date,
            day_ts=int(datetime.datetime.strptime(self.date, "%Y-%m-%d").strftime("%s")),
            # input tables
            soup_yasoft=soup_table(ids.YANDEXUID, ids.DISTR_UI, source_type.YASOFT, log_source.EXPORT_ACCESS_LOG),
            soup_yasoft_punto=soup_table(
                ids.YANDEXUID, ids.DISTR_UI, source_type.YASOFT_PUNTO_SWITCHER, log_source.EXPORT_ACCESS_LOG
            ),
            soup_external_bro=soup_table(
                ids.YANDEXUID, ids.DISTR_UI, source_type.YABRO_EXTERNAL_BROWSERS, log_source.EXPORT_ACCESS_LOG
            ),
            # output tables
            out_eal=self.out_f("raw") + "yuid_with_" + config.ID_SOURCE_TYPE_EAL,
            out_punto=self.out_f("raw") + "yuid_with_" + config.ID_SOURCE_TYPE_PUNTO,
            out_ext_bro=self.out_f("raw") + "yuid_with_" + config.ID_SOURCE_TYPE_EXTERNAL_BROWSERS,
        )

        with self.yt.Transaction() as tx:
            run_yql(query=template.format(tx=tx.transaction_id, **context))

    def output(self):
        return super(ImportEalDayTask, self).output() + [
            yt_luigi.YtTarget(self.out_f("raw") + "yuid_with_" + config.ID_SOURCE_TYPE_EAL),
            yt_luigi.YtTarget(self.out_f("raw") + "yuid_with_" + config.ID_SOURCE_TYPE_PUNTO),
            yt_luigi.YtTarget(self.out_f("raw") + "yuid_with_" + config.ID_SOURCE_TYPE_EXTERNAL_BROWSERS),
        ]
