import luigi
from collections import defaultdict

from crypta.graph.soup.config.python import LOG_SOURCE as log_source
from crypta.graph.v1.python.lib.luigi import base_luigi_task, yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils.yql_utils import run_yql
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable


def get_dict_table(name):
    return config.GRAPH_YT_DICTS_FOLDER + name


def mkrec(uid, uid_type, id_value, id_type, source_type, date, history_size):
    """Used in webvisor parser"""
    return {
        uid_type: uid,
        "id_value": id_value,
        "id_type": id_type,
        "source_type": source_type,
        "id_date": date,
        "id_count": history_size,
        "@table_index": 0,
    }


def reduce_uid_ids(uid_key, recs, uid_type):
    """Used in webvisor parser"""
    uid = uid_key[uid_type]
    # type -> (id -> count)
    ids_by_t = defaultdict(lambda: defaultdict(lambda: 0))

    id_date = ""
    for rec in recs:
        id_type = rec["id_type"]
        source_type = rec["source_type"]
        id_value = rec["id_value"]
        id_date = rec["id_date"]
        id_count = rec["id_count"]
        # To prevent OOM
        if len(ids_by_t[(id_type, source_type)]) < 100:
            ids_by_t[(id_type, source_type)][id_value] += id_count

    for (id_type, source_type), ids in ids_by_t.iteritems():
        for id_value, id_count in ids.iteritems():
            yield {
                uid_type: uid,
                "id_type": id_type,
                "source_type": source_type,
                "id_value": id_value,
                "id_date": id_date,
                "id_count": id_count,
            }

    ips = ids_by_t[(config.ID_TYPE_IP, config.ID_SOURCE_TYPE_FP)]
    for ip in ips.keys():
        yield {"ip": ip, uid_type: uid, "@table_index": 1}


class ImportFPDayTask(base_luigi_task.BaseTask):

    resources = {"import_fp_lock": 1}

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportFPDayTask, self).__init__(*args, **kwargs)
        self.soup_logs = [
            SoupDailyLogTable(log_source.PASSPORT_LOG, self.date),
            SoupDailyLogTable(log_source.REQANS_LOG, self.date),
        ]

    def requires(self):
        return [
            # required all external logs
            yt_luigi.ExternalInput(config.LOG_FOLDERS["bs_watch"] + self.date),
            yt_luigi.ExternalInput(config.LOG_FOLDERS["passport"] + self.date),
            yt_luigi.ExternalInput(config.LOG_FOLDERS["bar"] + self.date),
            yt_luigi.ExternalInput(config.LOG_FOLDERS["reqans_new"] + self.date),
            yt_luigi.ExternalInput(config.LOG_FOLDERS["reqans_alice"] + self.date, allow_empty=True),
        ]

    def run(self):
        for soup_log in self.soup_logs:
            soup_log.ensure_dir()
        run_yql("FingerPrintParser", dict(date=self.date))
        for soup_log in self.soup_logs:
            soup_log.prepare_daily_tables_from_log()

    def output(self):
        out_folder = config.YT_OUTPUT_FOLDER + self.date + "/"

        last_day_dependencies = []
        if self.date == self.run_date:
            for soup_log in self.soup_logs:
                last_day_dependencies.extend(soup_log.daily_tables_targets())

        return [
            yt_luigi.YtTarget(out_folder + "yuids_ua_day"),
            yt_luigi.YtTarget(out_folder + "yuids_desk_day"),
            yt_luigi.YtTarget(out_folder + "yuids_mob_day"),
            yt_luigi.YtTarget(out_folder + "yuid_raw/yuid_with_info"),
            yt_luigi.YtTarget(
                out_folder + "yuid_raw/yuid_with_" + config.ID_TYPE_LOGIN + "_" + config.ID_SOURCE_TYPE_FP
            ),
            yt_luigi.YtTarget(
                out_folder + "yuid_raw/yuid_with_" + config.ID_TYPE_VKCOM + "_" + config.ID_SOURCE_TYPE_FP
            ),
            yt_luigi.YtTarget(
                out_folder + "yuid_raw/yuid_with_" + config.ID_TYPE_OKRU + "_" + config.ID_SOURCE_TYPE_FP,
                allow_empty=True,
            ),
            yt_luigi.YtTarget(
                out_folder + "yuid_raw/yuid_with_" + config.ID_TYPE_PUID + "_" + config.ID_SOURCE_TYPE_FP
            ),
            yt_luigi.YtTarget(out_folder + "yuids_rus_ip_day"),
        ] + last_day_dependencies
