from functools import partial

import luigi

from crypta.graph.v1.python.data_imports.day_aggregate import (
    reduce_yuid_log_events_day,
    finalize_yuid_with_x_day_tables,
)
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable

from crypta.graph.soup.config.python import (
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


def map_passport_phone_md5(rec, puid_phonehash_psphn):
    phone = rec["phone"]
    id_prefix, id_hash = utils.prepare_phone_md5(phone)  # can't keep original phone from passport
    yield {
        "id_value": id_hash,
        "id_prefix": id_prefix,
        "id_type": config.ID_TYPE_PHONE,
        "source_type": config.ID_SOURCE_TYPE_PASSPORT,
        "key": rec["uid"],
        "puid": rec["uid"],
    }
    yield SoupDailyLogTable.make_rec(rec["uid"], id_hash, puid_phonehash_psphn, table_index=1)


def map_short_session(rec):
    if "ttl" in rec and rec["ttl"] == "0":  # short session according to https://wiki.yandex-team.ru/passport/statbox/
        yuid = rec.get("yandexuid")
        puid = rec.get("uid")
        login = rec.get("input_login")
        if yuid and puid and login:
            login = login.replace("@yandex.ru", "")  # TODO: find normalized login or make correct normalization
            yield {"yuid": yuid, "puid": puid, "id_value": login}


class ImportPassportPhoneDayTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportPassportPhoneDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.PASSPORT_PHONE_LOG, self.date)

    def input_folders(self):
        return {"passport_phone_log": config.LOG_FOLDERS["passport_phone"]}

    def output_folders(self):
        return {
            "passport": config.YT_OUTPUT_FOLDER + self.date + "/passport/",
            "yuid_raw_day": config.YT_OUTPUT_FOLDER + self.date + "/yuid_raw/",
        }

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f("passport_phone_log") + self.date)

    def before_run(self):
        out_yuid_raw = self.out_f("yuid_raw_day")
        out_passport = self.out_f("passport")

        mr.mkdir(out_yuid_raw)
        mr.mkdir(out_passport)
        self.soup_log.ensure_dir()

    def run(self):
        passport_phone_log = self.in_f("passport_phone_log") + self.date
        out_yuid_raw = self.out_f("yuid_raw_day")
        out_passport = self.out_f("passport")

        # extract from logfeller for today
        self.yt.run_map(
            partial(
                map_passport_phone_md5,
                puid_phonehash_psphn=edges.get_edge_type(
                    ids.PUID, ids.PHONE_MD5, source_type.PASSPORT_PROFILE, log_source.PASSPORT_PHONE_LOG
                ),
            ),
            passport_phone_log,
            [out_passport + "puid_md5phone_passport_day", self.soup_log.create()],
        )

        self.soup_log.prepare_daily_tables_from_log()

        self.yt.run_sort(out_passport + "puid_md5phone_passport_day", sort_by="puid")

        out_table = out_yuid_raw + "puid_with_" + config.ID_TYPE_PHONE + "_" + config.ID_SOURCE_TYPE_PASSPORT
        self.yt.run_reduce(
            partial(
                reduce_yuid_log_events_day,
                dt=self.date,
                id_type=config.ID_TYPE_PHONE,
                source_type=config.ID_SOURCE_TYPE_PASSPORT,
                key_col="puid",
            ),
            out_passport + "puid_md5phone_passport_day",
            out_table,
            reduce_by="puid",
        )

        finalize_yuid_with_x_day_tables([out_table], key_col="puid")

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        t = self.out_f("yuid_raw_day") + "puid_with_" + config.ID_TYPE_PHONE + "_" + config.ID_SOURCE_TYPE_PASSPORT
        return [yt_luigi.YtTarget(t)] + soup_out_tables


class ImportPassportLogDayTask(yt_luigi.BaseYtTask):
    """
    Only fetches short session metrics for now!
    Real passport logins are fetched from
    """

    date = luigi.Parameter()
    run_date = luigi.Parameter()
    tags = ["v1"]

    def input_folders(self):
        return {"passport_log": config.LOG_FOLDERS["passport"]}

    def output_folders(self):
        return {"passport": config.YT_OUTPUT_FOLDER + self.date + "/passport/"}

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f("passport_log") + self.date)

    def before_run(self):
        out_passport = self.out_f("passport")

        mr.mkdir(out_passport)

    def run(self):
        passport_log = self.in_f("passport_log") + self.date
        out_passport = self.out_f("passport")

        # short session == галочка чужой компьютер
        self.yt.run_map(map_short_session, passport_log, out_passport + "short_session_raw")
        mr.distinct_by(
            ["yuid", "puid", "id_value"], out_passport + "short_session_raw", out_passport + "short_session_raw"
        )

    def output(self):
        return [yt_luigi.YtTarget(self.out_f("passport") + "short_session_raw")]
