from functools import partial

import luigi
import json

from crypta.graph.v1.python.data_imports.day_aggregate import reduce_yuid_log_events_day
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable
from crypta.lib.python.identifiers.identifiers import Email

from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


def map_log(rec, dt, yuid_email_senderday):
    try:
        cookies = json.loads(rec.get("cookies"))
    except (ValueError, TypeError):
        cookies = {}

    email = rec.get("email")
    yuid = cookies.get("yandexuid")
    if yuid and email and Email(email).is_valid():
        yield {"id_value": email, "yuid": yuid, "id_date": dt}
        yield SoupDailyLogTable.make_rec(yuid, email, yuid_email_senderday, table_index=1)


class ImportSenderDayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportSenderDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.SENDER_LOG, self.date)

    def input_folders(self):
        return {"logfeller": config.LOGFELLER_SENDER_FOLDER}

    def output_folders(self):
        return {"graph": config.YT_OUTPUT_FOLDER + self.date + "/"}

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f("logfeller") + self.date)

    def before_run(self):
        mr.mkdir(self.out_f("graph") + "yuid_raw")
        self.soup_log.ensure_dir()

    def run(self):
        temporary_table = self.out_f("graph") + "yuid_raw/temporary_table_tickets"

        self.yt.run_map(
            partial(
                map_log,
                dt=self.date,
                yuid_email_senderday=edges.get_edge_type(
                    ids.YANDEXUID, ids.EMAIL, source_type.EMAIL_SENDER, log_source.SENDER_LOG
                ),
            ),
            self.in_f("logfeller") + self.date,
            [temporary_table, self.soup_log.create()],
        )

        self.soup_log.prepare_daily_tables_from_log()

        self.yt.run_map_reduce(
            None,
            partial(
                reduce_yuid_log_events_day,
                dt=self.date,
                id_type=config.ID_TYPE_EMAIL,
                source_type=config.ID_SOURCE_TYPE_SENDER,
            ),
            temporary_table,
            self.out_f("graph") + "yuid_raw/yuid_with_" + config.ID_SOURCE_TYPE_SENDER,
            sort_by="yuid",
            reduce_by="yuid",
        )

        mr.drop(temporary_table)

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        return [
            yt_luigi.YtTarget(
                self.out_f("graph") + "yuid_raw/yuid_with_" + config.ID_SOURCE_TYPE_SENDER, allow_empty=True
            )
        ] + soup_out_tables
