import json
from functools import partial

import luigi

from crypta.graph.v1.python.data_imports.day_aggregate import (
    reduce_yuid_log_events_day,
    finalize_yuid_with_x_day_tables,
)
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable

from crypta.graph.soup.config.python import (  # noqa
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


OOM_LIMIT = 100


def map_mitb(rec, yuid_ui_sbapimitblog):
    data = rec.get("data")
    if data and data != "-":
        try:
            json_data = json.loads(data.decode("string_escape"))
            ui = json_data["ui"].strip("{}")
            yuids_bro = dict()
            for yuid_json in json_data["yandexuids"]:
                yuid = yuid_json["value"]
                if yuid:
                    yuids_bro[yuid] = yuid_json["browser"]

            for yuid, browser in yuids_bro.iteritems():
                yield {"id_value": ui, "yuid": yuid, "browser": browser}
                yield SoupDailyLogTable.make_rec(yuid, ui, yuid_ui_sbapimitblog, table_index=2)

        except Exception:
            yield {"data": data, "@table_index": 1}


class ImportSbApiMitbLogDayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportSbApiMitbLogDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.SBAPI_MITB_LOG, self.date)

    def input_folders(self):
        return {"log": config.LOGFELLER_SBAPI_MITB_LOG_FOLDER}

    def output_folders(self):
        return {
            "sbapi_tmp": config.YT_OUTPUT_FOLDER + self.date + "/sbapi/",
            "yuid_raw_day": config.YT_OUTPUT_FOLDER + self.date + "/yuid_raw/",
        }

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f("log") + self.date)

    def before_run(self):
        mr.mkdir(self.out_f("sbapi_tmp"))
        mr.mkdir(self.out_f("yuid_raw_day"))
        self.soup_log.ensure_dir()

    def run(self):
        self.yt.run_map(
            partial(
                map_mitb,
                yuid_ui_sbapimitblog=edges.get_edge_type(
                    ids.YANDEXUID, ids.DISTR_UI, source_type.YASOFT_BROWSER_MANAGER, log_source.SBAPI_MITB_LOG
                ),
            ),
            self.in_f("log") + self.date,
            [
                self.out_f("sbapi_tmp") + "yuid_ui_mitb_log",
                self.out_f("sbapi_tmp") + "yuid_ui_mitb_log_errrs",
                self.soup_log.create(),
            ],
        )

        self.soup_log.prepare_daily_tables_from_log()

        out_table = "".join(
            (
                self.out_f("yuid_raw_day"),
                "yuid_with_",
                config.ID_TYPE_BAR_UI,
                "_",
                config.ID_SOURCE_TYPE_BROWSER_MANAGER,
            )
        )
        self.yt.run_map_reduce(
            None,
            partial(
                reduce_yuid_log_events_day,
                dt=self.date,
                id_type=config.ID_TYPE_BAR_UI,
                source_type=config.ID_SOURCE_TYPE_BROWSER_MANAGER,
            ),
            self.out_f("sbapi_tmp") + "yuid_ui_mitb_log",
            out_table,
            reduce_by=config.ID_TYPE_YUID,
        )

        finalize_yuid_with_x_day_tables([out_table])

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        return [
            yt_luigi.YtTarget(
                "".join(
                    (
                        self.out_f("yuid_raw_day"),
                        "yuid_with_",
                        config.ID_TYPE_BAR_UI,
                        "_",
                        config.ID_SOURCE_TYPE_BROWSER_MANAGER,
                    )
                )
            )
        ] + soup_out_tables
