import re
from functools import partial

import luigi

from crypta.graph.v1.python.data_imports.day_aggregate import (
    reduce_yuid_log_events_day,
    finalize_yuid_with_x_day_tables,
)
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable

from crypta.graph.soup.config.python import (
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


def map_log(rec, dt, yuid_vkcom, yuid_vkcom_name, yuid_okru, yuid_facebook_id):
    OK_TABLE_INDEX, FB_TABLE_INDEX, VK_TABLE_INDEX, VK_NAME_TABLE_INDEX, SOUP_TABLE_INDEX = range(5)

    id_sn = rec.get("id", "")
    if not id_sn:
        # fast break
        return
    type_sn = rec.get("type_sn", "").lower()
    yandexuid = rec.get("yandexuid", "")

    def vk_norm(val):
        """ Clean vk id from id\\d+ and switch vk_id and vk_name tables """
        match = re.match(r"^id(\d+)$", val)
        if match:
            # vk_id table
            return yuid_vkcom, VK_TABLE_INDEX, match.group(1)
        return yuid_vkcom_name, VK_NAME_TABLE_INDEX, val

    edge_choice = {
        config.ID_TYPE_OKRU.lower(): lambda val: (yuid_okru, OK_TABLE_INDEX, val),
        config.ID_TYPE_FACEBOOK_ID.lower(): lambda val: (yuid_facebook_id, FB_TABLE_INDEX, val),
        config.ID_TYPE_VKCOM.lower(): vk_norm,
    }

    try:
        edge_type, table_id, id_sn = edge_choice[type_sn](id_sn)
    except KeyError:
        return

    yield {"id_value": id_sn, "id_date": dt, "yuid": yandexuid, "@table_index": table_id}
    yield SoupDailyLogTable.make_rec(yandexuid, id_sn, edge_type, table_index=SOUP_TABLE_INDEX)


class ImportSovetnikDayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def input_folders(self):
        return {"sovetnik": config.SOVETNIK_INPUT_FOLDER}

    def output_folders(self):
        return {"graph": config.YT_OUTPUT_FOLDER + self.date + "/"}

    def __init__(self, *args, **kwargs):
        super(ImportSovetnikDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.SOVETNIK, self.date)

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f("sovetnik") + "/" + self.date)

    def before_run(self):
        mr.mkdir(self.out_f("graph") + "yuid_raw")
        self.soup_log.ensure_dir()

    def run(self):
        def get_temporary_table(id_type):
            return "{prefix}yuid_raw/temporary_table_sovetnik_{id_type}".format(
                prefix=self.out_f("graph"), id_type=id_type
            )

        def get_out_table(id_type):
            return "{prefix}yuid_raw/yuid_with_{id_type}_{id_source}".format(
                prefix=self.out_f("graph"), id_type=id_type, id_source=config.ID_SOURCE_TYPE_SOVETNIK
            )

        tables_by_id_type = [
            (id_type, get_temporary_table(id_type), get_out_table(id_type))
            for id_type in [
                config.ID_TYPE_OKRU,
                config.ID_TYPE_FACEBOOK_ID,
                config.ID_TYPE_VKCOM,
                config.ID_TYPE_VKCOM_NAME,
            ]
        ]

        self.yt.run_map(
            partial(
                map_log,
                yuid_vkcom=edges.get_edge_type(ids.YANDEXUID, ids.VK_ID, source_type.SOVETNIK, log_source.SOVETNIK),
                yuid_vkcom_name=edges.get_edge_type(
                    ids.YANDEXUID, ids.VK_NAME, source_type.SOVETNIK, log_source.SOVETNIK
                ),
                yuid_okru=edges.get_edge_type(ids.YANDEXUID, ids.OK_ID, source_type.SOVETNIK, log_source.SOVETNIK),
                yuid_facebook_id=edges.get_edge_type(
                    ids.YANDEXUID, ids.FB_ID, source_type.SOVETNIK, log_source.SOVETNIK
                ),
                dt=self.date,
            ),
            self.in_f("sovetnik") + "/" + self.date,
            [[tmp_table for _, tmp_table, _ in tables_by_id_type], self.soup_log.create()],
        )

        self.soup_log.prepare_daily_tables_from_log()

        utils.wait_all(
            [
                self.yt.run_map_reduce(
                    None,
                    partial(
                        reduce_yuid_log_events_day,
                        dt=self.date,
                        id_type=id_type,
                        source_type=config.ID_SOURCE_TYPE_SOVETNIK,
                    ),
                    temporary_table,
                    out_table,
                    sort_by="yuid",
                    reduce_by="yuid",
                    sync=False,
                )
                for id_type, temporary_table, out_table in tables_by_id_type
            ]
        )

        finalize_yuid_with_x_day_tables([out_table for _, _, out_table in tables_by_id_type])

        for _, temporary_table, _ in tables_by_id_type:
            mr.drop(temporary_table)

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []
        return [
            yt_luigi.YtTarget(
                "{prefix}yuid_raw/yuid_with_{id_type}_{id_source}".format(
                    prefix=self.out_f("graph"), id_type=config.ID_TYPE_OKRU, id_source=config.ID_SOURCE_TYPE_SOVETNIK
                ),
                allow_empty=True,
            ),
            yt_luigi.YtTarget(
                "{prefix}yuid_raw/yuid_with_{id_type}_{id_source}".format(
                    prefix=self.out_f("graph"),
                    id_type=config.ID_TYPE_FACEBOOK_ID,
                    id_source=config.ID_SOURCE_TYPE_SOVETNIK,
                ),
                allow_empty=True,
            ),
            yt_luigi.YtTarget(
                "{prefix}yuid_raw/yuid_with_{id_type}_{id_source}".format(
                    prefix=self.out_f("graph"), id_type=config.ID_TYPE_VKCOM, id_source=config.ID_SOURCE_TYPE_SOVETNIK
                ),
                allow_empty=True,
            ),
            yt_luigi.YtTarget(
                "{prefix}yuid_raw/yuid_with_{id_type}_{id_source}".format(
                    prefix=self.out_f("graph"),
                    id_type=config.ID_TYPE_VKCOM_NAME,
                    id_source=config.ID_SOURCE_TYPE_SOVETNIK,
                ),
                allow_empty=True,
            ),
        ] + soup_out_tables
