from textwrap import dedent
from datetime import datetime, timedelta

import luigi

from crypta.graph.v1.python.data_imports import StreamImportBaseTask, soup_table
from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils.yql_utils import run_yql

from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    LOG_SOURCE as log_source,
    SOURCE_TYPE as source_type,
)


class ImportWatchLogDayTask(StreamImportBaseTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()
    resources = {"import_watch_log_lock": 1}
    priority = 1

    observed_logs = ["bs-watch-log"]

    def __init__(self, *args, **kwargs):
        ls = log_source.WATCH_LOG
        wait_keys = ["WatchLogImportTask", "SoupTask"]
        super(ImportWatchLogDayTask, self).__init__(ls, wait_keys, *args, **kwargs)

    def output_folders(self):
        return {
            "yuid_raw": config.YT_OUTPUT_FOLDER + self.date + "/yuid_raw/",
            "devid_raw": config.INDEVICE_YT_FOLDER + self.date + "/perfect/devid_raw_day/",
            "ecommerce": config.YT_OUTPUT_FOLDER + self.date + "/ecommerce/",
        }

    def run_stream(self):

        # create dirs before transaction
        self.yt.mkdir(self.out_f("yuid_raw").rstrip("/"), recursive=True)
        self.yt.mkdir(self.out_f("devid_raw").rstrip("/"), recursive=True)
        self.yt.mkdir(self.out_f("ecommerce").rstrip("/"), recursive=True)

        def yuid_with(id_type, source_type):
            return "{}yuid_with_{}_{}".format(self.out_f("yuid_raw"), id_type, source_type)

        def appm_with(id_type, source_type):
            return "{}{}_yuid_{}".format(self.out_f("devid_raw"), id_type, source_type)

        template = dedent(
            """SELECT NULL; -- for python inline sql code highlight
                pragma yt.ExternalTx = '{tx}';

                -- soup stuff
                $yuid_email_pagetitle = '{yuid_email_pagetitle}';
                $yuid_mac_vmetro = '{yuid_mac_vmetro}';
                $yuid_purchase_tables = AsList('{yuid_purchase_tables}');
                $yuid_uuid_redir = '{yuid_uuid_redir}';
                $yuid_devid_ios = '{yuid_devid_ios}';
                $yuid_devid_android = '{yuid_devid_android}';
                $yuid_uuid_sockets_ios = '{yuid_uuid_sockets_ios}';
                $yuid_uuid_sockets_android = '{yuid_uuid_sockets_android}';
                $yuid_idfa_sockets = '{yuid_idfa_sockets}';
                $yuid_gaid_sockets = '{yuid_gaid_sockets}';

                -- v1 stuff
                $metrica_sockets_source_type_ios = '{metrica_sockets_source_type_ios}';
                $metrica_sockets_source_type_android = '{metrica_sockets_source_type_android}';

                $yuid_with_id_email = '{yuid_with_id_email}';
                $yuid_with_id_mac = '{yuid_with_id_mac}';
                $yuid_purchase_log = '{yuid_purchase_log}';
                $yuid_with_id_uuid = '{yuid_with_id_uuid}';
                $yuid_with_mm_devid_ios = '{yuid_with_mm_devid_ios}';
                $yuid_with_mm_devid_android = '{yuid_with_mm_devid_android}';
                $yuid_with_uuid_sockets_ios = '{yuid_with_uuid_sockets_ios}';
                $yuid_with_uuid_sockets_android = '{yuid_with_uuid_sockets_android}';
                $yuid_with_devid_sockets_ios = '{yuid_with_devid_sockets_ios}';
                $yuid_with_devid_sockets_android = '{yuid_with_devid_sockets_android}';
                $purchase_log_storage = '{purchase_log_storage}';
                $purchase_log_storage_cutoff_date = '{purchase_log_storage_cutoff_date}';

                $purchase_log = (
                    select * from EACH($yuid_purchase_tables)
                );

                insert into $purchase_log_storage with truncate
                select * from $purchase_log
                where dt >= $purchase_log_storage_cutoff_date;

                insert into $yuid_with_id_email with truncate
                select
                    id1 as yuid,
                    'email' as id_type,
                    'page_title' as source_type,
                    id2 as id_value,
                    '{dt}' as id_date,
                    1ul as id_count
                from $yuid_email_pagetitle
                where ListHas(dates, '{dt}')
                order by yuid;

                insert into $yuid_with_id_mac with truncate
                select
                    id1 as yuid,
                    'mac' as id_type,
                    'vmetro' as source_type,
                    String::ToLower(String::ReplaceAll(id2, ':', '')) as id_value,
                    '{dt}' as id_date,
                    1ul as id_count
                from $yuid_mac_vmetro
                where ListHas(dates, '{dt}')
                order by yuid;

                insert into $yuid_purchase_log with truncate
                select
                    yuid,
                    'purchase_domain' as id_type,
                    'ecommerce' as source_type,
                    id_value,
                    dt as id_date,
                    cast(count(1) as uint64) as id_count
                from $purchase_log
                where dt = '{dt}'
                group by yuid, id_value, dt as id_date
                order by yuid, id_value;

                insert into $yuid_with_id_uuid with truncate
                select
                    id1 as yuid,
                    id2 as `uuid`,
                    '{dt}' as dt,
                    1ul as hits_count,
                    -cast({end_day_ts} as int64) as last_ts,
                    'watch_log' as source_type
                from $yuid_uuid_redir
                where ListHas(dates, '{dt}')
                order by yuid, `uuid`, dt;

                insert into $yuid_with_mm_devid_ios with truncate
                select
                    id1 as yuid,
                    id2 as mmetric_devid,
                    '{dt}' as dt,
                    1ul as hits_count,
                    '{end_day_ts}' as last_ts,
                    'watch_yp_did_ios' as source_type
                from $yuid_devid_ios
                where ListHas(dates, '{dt}')
                order by yuid, mmetric_devid;

                insert into $yuid_with_mm_devid_android with truncate
                select
                    id1 as yuid,
                    id2 as mmetric_devid,
                    '{dt}' as dt,
                    1ul as hits_count,
                    '{end_day_ts}' as last_ts,
                    'watch_yp_did_android' as source_type
                from $yuid_devid_android
                where ListHas(dates, '{dt}')
                order by yuid, mmetric_devid;

                insert into $yuid_with_uuid_sockets_ios with truncate
                select
                    id1 as yuid,
                    id2 as `uuid`,
                    '{dt}' as dt,
                    1ul as hits_count,
                    -cast({end_day_ts} as int64) as last_ts,
                    $metrica_sockets_source_type_ios as source_type
                from $yuid_uuid_sockets_ios
                where ListHas(dates, '{dt}')
                order by yuid, `uuid`, dt;

                insert into $yuid_with_uuid_sockets_android with truncate
                select
                    id1 as yuid,
                    id2 as `uuid`,
                    '{dt}' as dt,
                    1ul as hits_count,
                    -cast({end_day_ts} as int64) as last_ts,
                    $metrica_sockets_source_type_android as source_type
                from $yuid_uuid_sockets_android
                where ListHas(dates, '{dt}')
                order by yuid, `uuid`, dt;

                insert into $yuid_with_devid_sockets_ios with truncate
                select
                    id1 as yuid,
                    id2 as devid,
                    '{dt}' as dt,
                    1ul as hits_count,
                    '{end_day_ts}' as last_ts,
                    $metrica_sockets_source_type_ios as source_type
                from $yuid_idfa_sockets
                where ListHas(dates, '{dt}')
                order by yuid, devid;

                insert into $yuid_with_devid_sockets_android with truncate
                select
                    id1 as yuid,
                    id2 as devid,
                    '{dt}' as dt,
                    1ul as hits_count,
                    '{end_day_ts}' as last_ts,
                    $metrica_sockets_source_type_android as source_type
                from $yuid_gaid_sockets
                where ListHas(dates, '{dt}')
                order by yuid, devid;
            """
        )

        with self.yt.Transaction() as tx:
            purchase_log_storage = "purchase_log_storage"
            # Store this many days of purchase log so that this import task could be rerun
            purchase_log_storage_days = 7

            purchase_log_stream_results = self.conf.paths.stream.extra_data + "/WatchLogImportTask/yuid_purchase_log"
            purchase_log_tables = self.yt.list(purchase_log_stream_results, absolute=True)

            context = dict(
                tx=tx.transaction_id,
                yuid_email_pagetitle=soup_table(
                    ids.YANDEXUID, ids.EMAIL, source_type.PAGE_TITLE, log_source.WATCH_LOG
                ),
                yuid_mac_vmetro=soup_table(ids.YANDEXUID, ids.MAC, source_type.VMETRO, log_source.WATCH_LOG),
                yuid_purchase_tables="', '".join(purchase_log_tables),
                yuid_uuid_redir=soup_table(ids.YANDEXUID, ids.UUID, source_type.APP_URL_REDIR, log_source.WATCH_LOG),
                yuid_devid_ios=soup_table(
                    ids.YANDEXUID, ids.MM_DEVICE_ID, source_type.WATCH_YP_COOKIE_IOS, log_source.WATCH_LOG
                ),
                yuid_devid_android=soup_table(
                    ids.YANDEXUID, ids.MM_DEVICE_ID, source_type.WATCH_YP_COOKIE_ANDROID, log_source.WATCH_LOG
                ),
                yuid_uuid_sockets_ios=soup_table(
                    ids.YANDEXUID, ids.UUID, source_type.APP_METRICA_SOCKETS_IOS, log_source.WATCH_LOG
                ),
                yuid_uuid_sockets_android=soup_table(
                    ids.YANDEXUID, ids.UUID, source_type.APP_METRICA_SOCKETS_ANDROID, log_source.WATCH_LOG
                ),
                yuid_idfa_sockets=soup_table(
                    ids.YANDEXUID, ids.IDFA, source_type.APP_METRICA_SOCKETS_IOS, log_source.WATCH_LOG
                ),
                yuid_gaid_sockets=soup_table(
                    ids.YANDEXUID, ids.GAID, source_type.APP_METRICA_SOCKETS_ANDROID, log_source.WATCH_LOG
                ),
                metrica_sockets_source_type_ios=config.ID_SOURCE_TYPE_METRICA_SOCKETS_IOS,
                metrica_sockets_source_type_android=config.ID_SOURCE_TYPE_METRICA_SOCKETS_ANDROID,
                yuid_with_id_email=yuid_with("email", "page_title"),
                yuid_with_id_mac=yuid_with("mac", "vmetro"),
                yuid_purchase_log=self.out_f("ecommerce") + "yuid_with_purchase_day",
                yuid_with_id_uuid=appm_with("uuid", "watch_log"),
                yuid_with_mm_devid_ios=appm_with("mmetric_devid", "watch_yp_did_ios"),
                yuid_with_mm_devid_android=appm_with("mmetric_devid", "watch_yp_did_android"),
                yuid_with_uuid_sockets_ios=appm_with("uuid", config.ID_SOURCE_TYPE_METRICA_SOCKETS_IOS),
                yuid_with_uuid_sockets_android=appm_with("uuid", config.ID_SOURCE_TYPE_METRICA_SOCKETS_ANDROID),
                yuid_with_devid_sockets_ios=appm_with("devid", config.ID_SOURCE_TYPE_METRICA_SOCKETS_IOS),
                yuid_with_devid_sockets_android=appm_with("devid", config.ID_SOURCE_TYPE_METRICA_SOCKETS_ANDROID),
                dt=self.date,
                end_day_ts=(
                    datetime.strptime(self.date, "%Y-%m-%d") + timedelta(hours=24) - datetime(1970, 1, 1)
                ).total_seconds(),
                purchase_log_storage=purchase_log_stream_results + "/" + purchase_log_storage,
                purchase_log_storage_cutoff_date=(
                    datetime.strptime(self.date, "%Y-%m-%d") - timedelta(days=purchase_log_storage_days)
                ).strftime("%Y-%m-%d"),
            )

            run_yql(query=template.format(**context))

            for t in purchase_log_tables:
                if not t.endswith(purchase_log_storage):
                    self.yt.remove(t)

    def output(self):
        return super(ImportWatchLogDayTask, self).output() + [
            yt_luigi.YtTarget(
                self.out_f("yuid_raw") + "yuid_with_" + config.ID_TYPE_EMAIL + "_" + config.ID_SOURCE_TYPE_PAGE_TITLE
            ),
            yt_luigi.YtTarget(
                self.out_f("yuid_raw") + "yuid_with_" + config.ID_TYPE_MAC + "_" + config.ID_SOURCE_TYPE_VMETRO,
                allow_empty=True,
            ),  # not available for all 30 days yet
            yt_luigi.YtTarget(self.out_f("devid_raw") + "uuid_yuid_" + config.ID_SOURCE_TYPE_WATCH_LOG),
            yt_luigi.YtTarget(self.out_f("ecommerce") + "yuid_with_purchase_day"),
        ]


class WaitWatchLogSoup(StreamImportBaseTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    observed_logs = ["bs-watch-log"]

    def __init__(self, *args, **kwargs):
        ls = log_source.WATCH_LOG
        wait_keys = ["WatchLogImportTask", "SoupTask"]
        super(WaitWatchLogSoup, self).__init__(ls, wait_keys, *args, **kwargs)
