import luigi
import yt.wrapper as yt

from crypta.graph.v1.python.lib.luigi import yt_luigi
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils.yql_utils import run_yql
from crypta.graph.v1.python.v2.soup.soup_tables import SoupDailyLogTable

from crypta.graph.soup.config.python import (  # N811 # noqa
    ID_TYPE as ids,
    SOURCE_TYPE as source_type,
    LOG_SOURCE as log_source,
    EDGE_TYPE as edges,
)


def get_yid_raw_table(date, id_type):
    return yt.ypath.ypath_join(
        config.YT_OUTPUT_FOLDER,
        date,
        "yuid_raw",
        "yuid_with_{id_type}_{id_source}".format(
            id_type=id_type,
            id_source=config.ID_SOURCE_TYPE_WEBVISOR
        )
    )


class ImportWebvisorDayTask(yt_luigi.BaseYtTask):

    """
    Import webvisor 2+ from dayli parsed tables
    """

    date = luigi.Parameter()
    resources = {"import_webvisor_lock": 1}
    priority = 1
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportWebvisorDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(log_source.WEBVISOR_LOG, self.date)

    def requires(self):
        return [yt_luigi.ExternalInput(self.in_f("webvisor") + self.date)]

    def input_folders(self):
        return {"webvisor": config.LOG_FOLDERS["bs_webvisor_log"]}

    def output_folders(self):
        return {
            "webvisor_processed": yt.ypath.ypath_join(config.WEBVISOR_LOGIN_FOLDER_PROCESSED, self.date),
            "yuid_raw_email": get_yid_raw_table(self.date, config.ID_TYPE_EMAIL),
            "yuid_raw_phone": get_yid_raw_table(self.date, config.ID_TYPE_PHONE),
            "soup_table": self.soup_log.table_path(),
        }

    def before_run(self):
        mr.mkdir(config.WEBVISOR_LOGIN_FOLDER_PROCESSED)

        self.soup_log.ensure_dir()

        # TODO: added schema for webvisor_processed and yuid_raw tables

    def run(self):
        params = dict(
            date=self.date,
            webvisor_processed=self.out_f("webvisor_processed"),
            yuid_raw_email=self.out_f("yuid_raw_email"),
            yuid_raw_phone=self.out_f("yuid_raw_phone"),
            soup_table=self.out_f("soup_table"),
            webvisor_tolerance=config.WEBVISOR_TOLERANCE,
        )

        run_yql("WebVisorParser", params)

        self.soup_log.prepare_daily_tables_from_log()

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        return [
            yt_luigi.YtTarget(self.out_f("yuid_raw_email"), allow_empty=False),
            yt_luigi.YtTarget(self.out_f("yuid_raw_phone"), allow_empty=False),
        ] + soup_out_tables
