import datetime
import logging
import sys

import luigi
import yt.wrapper as yt

from crypta.graph.v1.python.lib import yt_trace
from crypta.graph.v1.python.lib.crypta_api import report_task_status_to_api
from crypta.graph.v1.python.rtcconf import config
from crypta.graph.v1.python.utils import logging_setup
from crypta.graph.v1.python.utils import mr_utils as mr
from crypta.graph.v1.python.utils import utils
from crypta.graph.v1.python.v2.soup import soup_dirs
from crypta.graph.v1.python import graph_v1
from crypta.graph.v1.python import graph_v2


class GraphPostprocTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        return [
            graph_v1.GraphV1PostprocTask(date=self.date, name="postproc-v1"),
            graph_v2.GraphV2PostprocTask(date=self.date, name="postproc-v2"),
        ]


class MainTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        return [
            # these dependencies are required also for luigi.run() initialization
            graph_v1.MainV1Task(date=self.date),
            graph_v2.MainV2Task(date=self.date),
        ]


@luigi.Task.event_handler(luigi.Event.FAILURE)
def celebrate_failure(task, e):
    utils.monrun_luigi_error(task, e)
    report_task_status_to_api(task, "FAILURE")


@luigi.Task.event_handler(luigi.Event.PROCESS_FAILURE)
def on_process_failure(task, e):
    utils.monrun_luigi_error(task, e)
    report_task_status_to_api(task, "FAILURE")


@luigi.Task.event_handler(luigi.Event.DEPENDENCY_MISSING)
def on_missing_dependency(task):
    logging.warn("Missing dependency during execution: %s" % task)


@luigi.Task.event_handler(luigi.Event.SUCCESS)
def mourn_success(task):
    utils.monrun_luigi_ok(task)
    report_task_status_to_api(task, "SUCCESS")


def main(luigi_task, date=None):

    # retry external tables and keep workers alive while idle for at most 28h
    luigi.worker.worker.retry_external_tasks = luigi.BoolParameter(default=True)
    luigi.worker.worker.keep_alive = luigi.BoolParameter(default=True)
    luigi.worker.worker.max_keep_alive_idle_duration = luigi.TimeDeltaParameter(default=datetime.timedelta(hours=28))
    luigi.scheduler.scheduler.retry_delay = luigi.IntParameter(default=10 * 60)

    yt.config.set_proxy(config.MR_SERVER)
    yt.config["tabular_data_format"] = yt.YsonFormat(control_attributes_mode="row_fields")
    yt.config.INTERMEDIATE_DATA_ACCOUNT = "crypta"

    # default max row weight 32mb
    yt.config["table_writer"] = {"max_row_weight": 32 << 10 << 10}

    logging_setup.setup_root_logger()
    logging_setup.setup_luigi_logging()
    logging_setup.setup_yt_logging()

    if date is None:
        dt = sys.argv[1]
    else:
        dt = date

    mr.mkdir(config.GRAPH_YT_DICTS_FOLDER)
    mr.mkdir(config.YT_OUTPUT_FOLDER + dt + "/mobile/")
    mr.mkdir(config.YT_OUTPUT_FOLDER + dt + "/raw_links/")
    mr.mkdir(config.INDEVICE_YT_FOLDER + dt + "/perfect/")

    mr.mkdir(soup_dirs.SOUP_DIR)
    mr.mkdir(soup_dirs.SOUP_PREPROCESS_DIR)
    mr.mkdir(soup_dirs.SOUP_DAY_DIR)
    mr.mkdir(soup_dirs.SOUP_DUMPS_DIR)
    mr.mkdir(soup_dirs.SOUP_DAY_LOGS_DIR)
    mr.mkdir(soup_dirs.SOUP_DAY_LOGS_DIR + "processing_status")

    yt_trace.setup_trace()

    retry_attempt = 0
    while retry_attempt < 3:
        if retry_attempt == 0:
            logging.info("Running luigi MainTask...")
        else:
            logging.info("Luigi failed. MainTask %d retry..." % retry_attempt)

        run_result = luigi.run(
            [
                luigi_task,
                "--date",
                date,
                "--workers",
                str(config.LUIGID_WORKERS),
                "--scheduler-url",
                config.LUIGID_URL,
                "--take-lock",
            ],
            detailed_summary=True,
        )

        if run_result.status in {luigi.LuigiStatusCode.SUCCESS, luigi.LuigiStatusCode.SUCCESS_WITH_RETRY}:
            utils.monrun_ok()
            sys.exit(0)
        else:
            logging.error(run_result.summary_text)
            retry_attempt += 1

    utils.monrun_error("Luigi finally failed after 3 attempts: ")
    sys.exit(1)


if __name__ == "__main__":
    main("MainTask")
