import luigi

from crypta.graph.v1.python.data_imports.import_logs import app_metrica_day
from crypta.graph.v1.python.data_imports.import_logs import graph_import_fp
from crypta.graph.v1.python.data_imports.import_logs.webvisor import graph_webvisor
from crypta.graph.v1.python.infra import graph_history_snapshot
from crypta.graph.v1.python.infra import graph_postproc
from crypta.graph.v1.python.infra import merge_graph_tables
from crypta.graph.v1.python.lib.luigi.yt_luigi import PostGraphTask
from crypta.graph.v1.python.matching import graph_resulted_pairs_tables
from crypta.graph.v1.python.matching.common_stats import graph_stat_report
from crypta.graph.v1.python.matching.device_matching.app_metrica import app_metrica_month
from crypta.graph.v1.python.matching.device_matching.distr import device_yuid_distr_dicts
from crypta.graph.v1.python.matching.device_matching.distr import yuid_distr_ui_desktop
from crypta.graph.v1.python.matching.human_matching import graph_vertices
from crypta.graph.v1.python.matching.human_matching.graph_clustering import ClusterVertices, ClusteringConfig
from crypta.graph.v1.python.matching.pairs.stats import graph_pairs_sources_stat
from crypta.graph.v1.python.matching.pairs.stats import graph_stat_passport
from crypta.graph.v1.python.matching.yuid_matching import graph_dict
from crypta.graph.v1.python.matching.yuid_matching.stats import graph_stat_id
from crypta.graph.v1.python.rtcconf import config


class GraphV1Tasks(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        exact_vertices_task = graph_vertices.GraphVerticesExact(
            self.date, vertices_type="exact", yuid_pairs_folder="pairs/"
        )
        exact_cluster = ClusterVertices(
            exact_vertices_task.vertices_config, ClusteringConfig(local_clustering_enabled=False)
        )

        # clustering experiment
        experiment_clustering_config = ClusteringConfig(new_multi_source_approach=True)
        exact_cluster_experiment = ClusterVertices(
            exact_vertices_task.vertices_config,
            clustering_config=experiment_clustering_config,
            name="cluster_experiment",
        )

        all_vertices_tasks = [exact_vertices_task, exact_cluster, exact_cluster_experiment]

        main_vertices_config = None
        to_bb = []

        for task in all_vertices_tasks:
            vertices_config = task.vertices_config
            if vertices_config.vertices_type in config.VERTICES_EXPERIMENTS:
                vertices_config.bb_experiment = True
                to_bb.append(vertices_config)

                if not config.VERTICES_EXPERIMENTS[vertices_config.vertices_type]:  # no experiment_id means production
                    if main_vertices_config:
                        raise Exception(
                            "You can't use two types of vertices in prod. "
                            "Conflict %s vs %s" % (main_vertices_config.vertices_type, vertices_config.vertices_type)
                        )
                    vertices_config.main_vertices = True
                    main_vertices_config = vertices_config

        if not main_vertices_config:
            raise Exception("At least one production vertices should be set")

        stats_tasks = [
            graph_stat_report.MatchingCoverageReportTask(
                date=self.date, name="statface-exact-done", vertices_config=exact_cluster.vertices_config
            ),
            graph_stat_passport.LoginShortSessionPairsStat(self.date),
            graph_stat_id.IdStatTask(date=self.date, name="id_stats"),
            graph_stat_id.YuidStatTask(self.date),
            graph_pairs_sources_stat.YuidIdBySourceStatsTask(date=self.date),
        ]

        crypta_id1_tasks = [
            main_vertices_config.producing_task,
            graph_import_fp.ImportFPDayTask(date=self.date, run_date=self.date),
            graph_webvisor.ImportWebvisorDayTask(date=self.date, run_date=self.date),
            app_metrica_day.ImportAppMetrikaDayTask(date=self.date, run_date=self.date),
            graph_vertices.CopyVerticesToDict(self.date, exact_task=exact_cluster),
        ]

        distr_tasks = [
            yuid_distr_ui_desktop.UiYuidAllTask(self.date),
            app_metrica_month.UpdateUuidDevidIndeviceAllDict(self.date),
            device_yuid_distr_dicts.UpdateYuidDevidIndeviceAllDict(self.date),
        ]

        other_consumers_tasks = [
            graph_dict.YuidWithIdXHash(self.date),
            graph_resulted_pairs_tables.CreateImportantPairsTablesAfterMatching(self.date, exact_task=exact_cluster),
        ]

        return stats_tasks + crypta_id1_tasks + distr_tasks + other_consumers_tasks


class GraphV1PostprocTask(PostGraphTask):
    def __init__(self, date, name="v1"):
        super(GraphV1PostprocTask, self).__init__(date=date, name=name)

    def run_post_graph(self):
        graph_postproc.run_postproc(safe_mode=False, run_date=self.date, clean_v1=True, clean_v2=False)
        merge_graph_tables.merge_v1_dicts()

    def requires(self):
        return GraphV1Tasks(date=self.date)


class MainV1Task(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        graph_all_task = GraphV1Tasks(date=self.date)
        tasks = [graph_all_task, GraphV1PostprocTask(date=self.date, name="postproc-v1")]
        if config.CRYPTA_ENV == "production":
            tasks.append(graph_history_snapshot.GraphHistorySnapshot(date=self.date, parent_task=graph_all_task))

        return tasks
